/*
  VSCHED: a scheduling tool for virtual machines
  http://virtuoso.cs.northwestern.edu
  (c) 2004 Bin Lin and Peter A. Dinda
*/

#include <iostream>
#include <strstream>
#include <string>
#include <vector>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/mman.h>
#include <iomanip>
#include <signal.h>
#include <sched.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <math.h>

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/* according to X/OPEN we have to define it ourselves */
union semun {
  int val;                  /* value for SETVAL */
  struct semid_ds *buf;     /* buffer for IPC_STAT, IPC_SET */
  unsigned short *array;    /* array for GETALL, SETALL */
  /* Linux specific part: */
  struct seminfo *__buf;    /* buffer for IPC_INFO */
};
#endif

//
// These headers provide the interface to the admission control system
// and to the scheduling core
//
// Both
//

#include "config.h"
#include "socks.h"
#include "admit.h"
#include "sched.h"

//SSL specific include libraries
#include <openssl/ssl.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/err.h>
#include <openssl/rsa.h>

// to obtain encryption details, turn this to 2
#define DETAILS 1

//Some definations for certificate and key file locations
#define HOME "./"
#define certfile HOME "vsched_cert.pem"
#define keyfile HOME "vsched_key.pem"

//Inititial SSL declarations

//VSCHED when as server
SSL *ssl1 = NULL;
SSL_CTX *ssl_ctx1 = NULL;

//The client and server certificates
X509 *server_cert;
X509 *client_cert;

//This is to indicate if we wish to use SSL (1) or not (0)
int use_ssl=0;


/* VSCHED configuration
   The scheduler process never looks at this
   it's only current in the parent
*/
string vsched_password;
string vsched_version("0.9");
string bind_address;
short  bind_port;
int    accept_socket;
bool	 FIFO_schedule;
int		 gui_prio, sched_prio;
int vindex;
struct sched_param mysched;

// This is a large structure containing all the information
// that is shared between the parent process (client handler) 
// and the child process (the scheduler)
// We will mmap this and then use the semaphore to
// synchronize access to it between parent and child
VSchedConfig   *config;
int            configsem;


void cleanup()
{
	exit(-1);
}

//------------------------------------------------------------------------------
// calculates an accurate value of sleep time and also an accurate value of LOOPCOUNT
// equal to that sleep time
//---------------------------------------------------------------------------------

// SIGCHLD handler
void Reaper(int s) 
{
  while(waitpid(-1,0,WNOHANG)>0)
    {
    }
  exit(-1);
}

#define GET(fd,ssl,s) if (GetLine(fd,ssl,s)<0) {close(fd);goto leave_fail2;} 
#define PUT(fd,ssl,s) if (PutLine(fd,ssl,s)!=(int)((s).size())) {goto leave_fail2;}

int HandleControlSession(const int afd, const int cfd, const struct sockaddr_in &adx, SSL *ssl1)
{
  string input, action, version, device, password;
  string output;
  
  	cout << "server is waiting for input" << endl;
  	GET(cfd,ssl1,input);
  	{
    	istrstream is(input.c_str(),input.size());
    	is >> action >> password >> version;
  	}

		cout << "message from client" << endl;
  	
  	if (action!="HELLO_VSCHED")
    { 
      output="NOK bad protocol";
      goto leave_error2;
    }
  	if (password!=vsched_password)
    { 
      output="NOK bad password";
      goto leave_error2;
    }
  	output="OK "+vsched_version+" continue" + "\n";
  	PUT(cfd,ssl1,output); 
  	
  	while (1)
    {
      GET(cfd,ssl1,input);
      {
        istrstream is(input.c_str(),input.size());
        is >> action;
      }
      
      cout << "client request: " << action << endl;
      
      if (action=="done")
        {
          output="OK disconnected";
          goto leave_ok2;
        }
      if (action=="config")
        {
          output="OK scheduler configuration";
          char buf[1024];
          
	  			config->Lock();
					
					snprintf(buf,1024,"FIFO: %d",config->FIFO);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"prio: %d",config->prio);
	  			output = output + "\n" + buf;
	 				snprintf(buf,1024,"numprocs: %d",config->numprocs);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"semaphore: %d",config->semaphore);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"valid: %d",config->valid);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"have_hires: %d",config->have_hires);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"period_min: %d",config->period_min);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"period_max: %d",config->period_max);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"period_res: %d",config->period_res);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"slice_min: %d",config->slice_min);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"slice_max: %d",config->slice_max);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"slice_res: %d",config->slice_res);
	  			output = output + "\n" + buf;
	  			snprintf(buf,1024,"sched_other_percent: %lf",config->sched_other_percent);
	  			output = output + "\n" + buf;
	 		 		snprintf(buf,1024,"allocated_time_percent: %lf",config->allocated_time_percent);
	  			output = output + "\n" + buf;
	  			
	 				config->Unlock();
	 				 
          output = output + "\n" + "\0";
          PUT(cfd, ssl1,output); 
          continue;
        }
      string temp;
      if (action=="ps")
        {
          output="NOK feature not implemented yet";
          PUT(cfd, ssl1,output); 
          continue;
        }
      if (action=="processes")
        {
          output="OK vsched-scheduled process list";
          char buf[1024];
          
          config->Lock();
          
          for(int iter = 0; iter <= config->numprocs; iter++)
          //for(int iter = 0; iter < 3; iter++)
					{
          	if(config->process[iter].valid)
						{	
          		snprintf(buf,1024,"valid: %d",config->process[iter].valid);
	  					output = output + "\n" + buf;
	  					snprintf(buf,1024,", pid: %d",config->process[iter].pid);
	  					output = output + buf;
	  					snprintf(buf,1024,", period_ms: %lf",config->process[iter].period_ms);
	  					output = output + buf;
	  					snprintf(buf,1024,", slice_ms: %lf",config->process[iter].slice_ms);
	  					output = output + buf;
	  					snprintf(buf,1024,", sched_admit_time: %d",config->process[iter].sched_admit_time);
	  					output = output + buf;
	  					snprintf(buf,1024,", suspended: %d",config->process[iter].suspended);
	  					output = output + buf;
	  					snprintf(buf,1024,", run_type: %s", (config->process[iter].run_type==FIFO ? "FIFO" : "OTHER"));
	  					output = output + buf;
	  					snprintf(buf,1024,", pathname: %s",config->process[iter].pathname);
	  					output = output + buf;
	  					snprintf(buf,1024,", next_deadline: %lf", config->process[iter].next_deadline);
	  					output = output + buf;
	  					snprintf(buf,1024,", left: %lf", config->process[iter].left);
	  					output = output + buf;	  						  					
	  					snprintf(buf,1024,", priority: %d", config->process[iter].priority);
	  					output = output + "\n" + buf;
	  				}
					}
					
          config->Unlock();
          
          output = output + "\n" + "\0";
          PUT(cfd, ssl1,output); 
          continue;
        }

      if (action=="schedule")
        {
	  			int pid;
	  			double period, slice;
	  
	  			istrstream is(input.c_str(),input.size());
          is >> action >> pid >> period >> slice;
	  
	  			//cout << action << pid << period << slice;
	  			
	  			double startup_time = 0.0;
	  			struct timeval tv;
	  			
	  			gettimeofday(&tv, NULL);
  				startup_time = (double)(tv.tv_sec)+(double)(tv.tv_usec)/1e6;
  				cout << "startup_time for new job is " << setiosflags(ios::fixed | ios:: showpoint) << startup_time << endl;
	  			
	  			VSchedProcess p;
	  			p.Init();
	  			p.pid=pid;
	  			p.period_ms = period;
	  			p.slice_ms = slice;
	  			p.left = p.slice_ms / 1000.0;
	  			p.next_deadline = startup_time + p.period_ms / 1000.0;
					p.valid=true;
					
	  			bool rc=AdmitIfPossible(*config,p);
	  			
	  			if (!rc) 
	  			{ 
	    			output="NOK cannot admit process";
	    			PUT(cfd, ssl1,output); 
	    			continue;
	  			} 
	  			else 
	  			{
	    			output="OK process admitted";
	    			//cout << "process list is updated.." << endl;	
	    			config->Set(p);
	    			cout << "notify scheduler by pipe, write: "<< write(config->pp[1], "SCHEDULE\n", 10) << endl; //tell scheduler that new job arrives
							    			
	    			PUT(cfd, ssl1, output);
	    			continue;
	  			}
				}

      if (action=="detach")
        { 
          int pid;
          istrstream is(input.c_str(),input.size());
          is >> action >> pid;
	  			
	  			VSchedProcess p;
	  			p.Init();
	  			p.pid=pid;
	  			p.valid=false;
	  			p.next_deadline=2147483648.0; //2^31
	  			
	  			//cout << "process list is updated.." << endl;
	  			config->Set(p);
	  			cout << "notify scheduler by pipe, write: "<< write(config->pp[1], "SCHEDULE\n", 10) << endl; //tell scheduler that new job arrives
          
          config->Lock();
          //unset FIFO first
          struct sched_param sched;
					sched.sched_priority = 0; //99, max
					if(sched_setscheduler(pid, SCHED_OTHER, &sched) == -1 )  //can be replaced by rt_task_init_schmod()
					{
							cout << "ERROR IN SETTING THE SCHEDULER DOWN" << endl;
							output = "NOK detach process fails";
							//->need to set run_type = SCHED_OTHER;
					}
					
					//set SCHED_OTHER priority
					int priority = 0;
					int ret;
					ret = setpriority(PRIO_PROCESS, pid, priority);
          
          //resume running
          if(kill(pid,SIGCONT) == -1)
					{
							cout << "error in resuming process: " << pid << " under SCHED_OTHER" << endl;
							output="NOK detach process fails";
					}
					else	    			
	  					output="OK process detached";
	  			
	  			config->Unlock();
	  					
          PUT(cfd,ssl1,output); 
          continue;
        }
        
      if (action=="stop")
        {
	  			int pid;
          istrstream is(input.c_str(),input.size());
          is >> action >> pid;

    			if(!config->GetPid2(pid))
    			{
    				cout << "current job can not be found in the list or it's set as not valid" << endl;
						output="NOK suspend process fails";
    			}
    			else
    			{	
          	
         		cout << "notify scheduler by pipe, write: "<< write(config->pp[1], "SCHEDULE\n", 10) << endl; //tell scheduler that new job arrives
          	
          	/*
          	if(kill(pid,SIGSTOP) == -1)
						{
								cout << "error in suspending process: " << pid << endl;
								output="NOK suspend process fails";
						}
						else	*/    			
	  						
	  				output="OK process suspended";
	  				
	  			}
	  			
	  			PUT(cfd, ssl1,output);
	  			continue;
				}

      if (action=="resume")
        {
	  			int pid;
          istrstream is(input.c_str(),input.size());
          is >> action >> pid;
          
          vindex = config->GetPid(pid);
    				
    			if(vindex < 0)
    			{
    				cout << "current job can not be found in the list or it's set as not valid" << endl;
						output="NOK resume process fails";
    			}
    			else
    			{	
          	config->Lock();
          	config->process[vindex].suspended = false;
          	config->Unlock();
          
         		cout << "notify scheduler by pipe, write: "<< write(config->pp[1], "SCHEDULE\n", 10) << endl; //tell scheduler that new job arrives    			
	  				output="OK process will resume running next time it gets scheduled by vsched.";
					}
				
	  			PUT(cfd, ssl1,output);
	  			continue;
				}
      
      // Unknown command
      output="NOK bad request";
      PUT(cfd, ssl1,output); 
    }

leave_fail2:
  close(cfd);
  if (use_ssl == 1)
    {
      SSL_shutdown(ssl1);
      if(ssl1 != NULL)
        free(ssl1);
    }
  return -1;
    
leave_ok2:
  PUT(cfd, ssl1,output);
  cout << "connection is closed" << endl;
  close(cfd); 
  if (use_ssl == 1)
    {
      SSL_shutdown(ssl1);
      if(ssl1 != NULL)
        free(ssl1);
    }
   return 0; 
  
leave_error2:
  PUT(cfd, ssl1,output);
  cout << output << endl;
  close(cfd);
  if (use_ssl == 1)
    {
      SSL_shutdown(ssl1);
      if(ssl1 != NULL)
        free(ssl1);
    }
  return -1;
	
}

int main(int argc, char *argv[])
{

  double percent_other;
  int numproc;
  int c;
  extern char *optarg;
  char *str;

#ifdef _KURT
	//************ begin of kurt *******************//	
	cout << "loading kurt..." << endl;
	int kurtdev;
	struct rtparams rt_param;

	/* Open /dev/kurt */
	kurtdev = kurt_open();

	/* set the scheduling mode to be KURT_ANYTIME
	 * and then set the rt_id of this procss properly
	 */
	rt_param.rt_id = ASSIGN_RT_ID;
	rt_param.period = 0;
	#ifdef CONFIG_RT_ENFORCEMENT
	rt_param.exec_time = 0;
	#endif
	rt_param.rt_mode = KURT_ANYTIME;
	rt_param.rt_name[0] = '\0';
	#ifdef CONFIG_SMP
	rt_param.processor = 0;
#endif
	
	if (set_rtparams(kurtdev, getpid(), RT_REGISTER, &rt_param) < 0) 
	{
		perror("set_rtparams");
		exit(-1);
	} 
	else 
	{
		get_rtparams(kurtdev, getpid(), &rt_param);
		cout << "KURT process rt_id:" << rt_param.rt_id << endl;
	}
	
	//*********** end of kurt ***********************//
#endif
		
  // defaults
  bind_port = DEFAULT_BINDPORT;
  bind_address = string(DEFAULT_BINDADDR);
  numproc = DEFAULT_NUMPROC;
  percent_other = DEFAULT_SCHEDOTHER;
	FIFO_schedule = false;
	gui_prio = 95;
	sched_prio = 98;
	
  if (argc<2)
  { 
      cerr << "usage: " << argv[0] <<  " password [-s] [-a] [-g <gui_prio>] [-e <sched_prio>] [-h <_|host>] [-p <port>] [-o <percentage>] [-n <numproc>]" << endl;
      return -1;
  }

  vsched_password = string(argv[1]);

  while ((c = getopt(argc, argv, "s a g:e:h:p:o:n:")) != -1)
    {
      switch(c)
        {
        case 's':
          use_ssl = 1;
          break;
        case 'a':
          FIFO_schedule = true;
          break;
        case 'g':
          gui_prio = atoi(optarg);
          break;
        case 'e':
          sched_prio = atoi(optarg);
          break;
        case 'h':
          bind_address = string(optarg);
          break;
        case 'p':
          bind_port = atoi(optarg);
          break;
        case 'o':
          percent_other=atof(optarg);
          break;
				case 'n':
	  			numproc=atoi(optarg);
        case '?':
	  			cerr << "usage: " << argv[0] <<  " password [-s] [-a] [-g <gui_prio>] [-e <sched_prio>] [-h <_|host>] [-p <port>] [-o <percentage>] [-n <numproc>]" << endl;
          exit(1);
        }
    }
			
  if (use_ssl == 1)
    {
      //Inititial SSL declarations
      SSL_load_error_strings();
      SSLeay_add_ssl_algorithms();
      
      ssl_ctx1 = SSL_CTX_new(SSLv23_server_method());
     
      //We next load the certificate
      if (SSL_CTX_use_certificate_file(ssl_ctx1, certfile, SSL_FILETYPE_PEM) <= 0)
        {
          cerr << "The local certificate could not be set from certfile" << endl;
          // exit(-1);
        }
      
      //We load the key
      if (SSL_CTX_use_PrivateKey_file(ssl_ctx1, keyfile, SSL_FILETYPE_PEM) <= 0)
        {
          cerr << "The private key could not be set from keyfile" << endl;
					cleanup();
        }
      
      //We check the key
      if (!SSL_CTX_check_private_key(ssl_ctx1))
        {
          cerr << "Private key does not match public certification" << endl;
          cleanup();
        }
    }

  // OK, now we'll create the memory for the config
  // This should give us an appopriately sized chunk of anonymous memory
  // that will automatically shared between parent and child.
  config = (VSchedConfig*) 
    mmap(0,
	 sizeof(VSchedConfig)+numproc*sizeof(VSchedProcess),
	 PROT_READ | PROT_WRITE,
	 MAP_ANONYMOUS | MAP_SHARED,
	 -1,
	 0);
  
  if ((int)config==-1) { 
    cerr << "Cannot allocate shared memory segment for configuration."<<endl;
    exit(-1);
  }

  configsem = semget(IPC_PRIVATE, 1, IPC_CREAT | 0600 );
  
  if (configsem==-1) { 
    cerr << "Cannot allocate semaphore for configuration segment."<<endl;
    cleanup();
  }

  // Now init it.
  union semun arg;
  arg.val=1;
  if (semctl(configsem,0,SETVAL,arg)==-1) { 
    cerr <<"Cannot initialize semaphore."<<endl;
    cleanup();
  }
  
  config->semaphore=configsem;
  config->Init(FIFO_schedule, sched_prio, numproc, percent_other);

  // OK, so set up signal handler in case child dies so that we die too.
  if (SetSignalHandler(SIGCHLD,Reaper)<0) {
      cerr << "Can't set SIGCHLD handler\n";
      cleanup();
  }

  // Now we will fork and have our child process 
  // configure itself as the scheduler,
  // implementing the requirements defined in config, which we will share

  int pid;
	
	if(pipe(config->pp) < 0)
	{
		cerr << "can't create a pipe" << endl;
  	cleanup();
	}
	
  pid=fork();

  if (pid < 0) 
  { 
    cerr << "Unable to fork." << endl;
    exit(-1);
  } 
  else if (pid==0) 
  { 
    //child process
    close(config->pp[1]);
    cout << "loading scheduler\n";
    Scheduler(*config);
  	cleanup();
  } 
  else 
  {
    // parent process
    mysched.sched_priority = gui_prio; //99, max
  	if( sched_setscheduler(0, SCHED_FIFO, &mysched ) == -1 )  //can be replaced by rt_task_init_schmod()
		{
			cout << "ERROR IN SETTING THE SCHEDULER UP" << endl;
			perror( "errno" );
			cleanup();
		}
		else
			cout << "set gui SCHED_FIFO with static priority " << mysched.sched_priority << endl;
			
		close(config->pp[0]);
    if ((accept_socket=CreateAndSetupTcpSocket())<0)      {
      cerr << "Can't setup socket\n";
      cleanup();
    }
    
    if (bind_address=="_")
    {
      if (BindSocket(accept_socket,bind_port)<0)
      { 
					cerr << "Can't bind socket\n";
					cleanup();
      }
    }    
    else
    {
      if (BindSocket(accept_socket,ToIPAddress(bind_address.c_str()),bind_port)<0)
			{
	  		cerr << "Can't bind socket\n";
	  		cleanup();
			}
    }
    
    if (ListenSocket(accept_socket)<0)
    {
      cerr << "Can't listen socket\n";
      cleanup();
    }
       
    struct sockaddr_in other;
    socklen_t salen;
    int connection_socket;
    
    // iterative server for control    
    while (1)      
    {
begin:
      salen=sizeof(other);
      
      connection_socket=accept(accept_socket, (struct sockaddr *) &other, &salen);
	
      if (connection_socket<0)
      {
					if (errno == EINTR)
					{
	  				continue;
					}	
					else	  
					{
	  				perror("Accept failed");
	  				cleanup();
					}
      }
      
      //At this point our TCP connection is setup and running
	
      if (use_ssl == 1)	  
      {
					//We will now be doing the SSL negotiation
					ssl1 = SSL_new(ssl_ctx1);
	
					SSL_set_fd(ssl1, connection_socket);
	
	
					int r;
					r = SSL_accept(ssl1);

					if (r < 0)	  
					{
	  					cerr << "SSL accept failed" << endl;
	  					string output2("SSL required");
	  					output2 = output2 + "\n";
	  					
	  					if (PutLine(connection_socket, NULL, output2)!=(int)((output2).size())) 
	  					{
	  							goto end;
	  					}
	  					else
	  							goto begin;
					}
	    
					//We will now print the encryption algorithm used
					if (DETAILS == 2) 
					{
	  					cout << "For the current client we are using the following encription: " << SSL_get_cipher(ssl1) << endl << endl;
					}
	
					//We next get the client's certificate and print the same to screen
					client_cert = SSL_get_peer_certificate(ssl1);
					if (client_cert == NULL)	      
					{
	  				if (DETAILS == 2)
	    					cout << "There are no client certificates" << endl;
					}	
					else	      
					{
	  				if (DETAILS == 2) 
	  				{
	    					cout << "Client Certificate:" << endl;
	  				}
	  				str = X509_NAME_oneline(X509_get_subject_name(client_cert),0,0);
	  				if (DETAILS == 2) 
	  				{
	    					cout << "    Subject: " << str << endl;
	  				}
	  				strcpy(str, "");
	  				str = X509_NAME_oneline(X509_get_issuer_name(client_cert),0,0);
	  				if (DETAILS == 2) 
	  				{
	    					cout << "    Issuer: " << str << endl;
	  				}
	  				free(str);
	  
	  				//This is where we can do all the server certificate verification stuff
	  				//That has not been added still except the one below
	  
	  				if(SSL_get_verify_result(ssl1) != X509_V_OK)	    
	  				{
	    					cerr << "The certificate does not verify" << endl;
	    					// Take appropriate action based on policy required
	  				}
					}
					//Now we are all set to move on to handling clients
					HandleControlSession(accept_socket, connection_socket,other,ssl1);
      
  			}  
  			else     
  			{
					// Handling the clients without SLL
					HandleControlSession(accept_socket, connection_socket,other,NULL);
	
      	}
   			
    }//end while(1)
 
end:
    
    close(connection_socket);
    if (use_ssl == 1)    
    {
      //We will now perform some cleaning up operatings and then leave
      SSL_shutdown(ssl1);
      if(ssl1 != NULL) 
      {
				free(ssl1);
      }
      if (ssl_ctx1 != NULL) 
      {
					free (ssl_ctx1);
      }
    }
   
    cleanup();
  }// end parent process
}

	

  
  
  
