#include "getloadavg.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <linux/types.h>
#include <errno.h>


/* FLAGS 
 * -i interval; default: 10 seconds
 * -l level; one/five/fifteen
 * -t transport; arp/tcp/ip
 *
 */

typedef enum LDAVG_LEVEL {ONE, FIVE, FIFTEEN} LDAVG_LEVEL;
typedef enum TRANSPORT {ARP, IP, TCP} TRANSPORT;

#define IPDSENDFILE "/proc/ipdsend"

int quiet_mode = 0;


void queue_data(double avg, __u32 addr, int mask, TRANSPORT t);
__u32 addr_char_to_u32(char * address);

void help() {
	printf("ldavgd\n");
	printf("usage: ldavgd [-i interval] [-l level] [-t transport] ipaddress\n\n");
	printf("\ti interval : interval between checking the avgs\n");
	printf("\tl level : which avg to send (1|5|15)\n");
	printf("\tt transport : which transport mechanism to use (ARP|TCP|IP)\n");
	exit(0);
}

int main(int argc, char ** argv) {
	int c, i = 0;
	int sleep_interval = 10;
	LDAVG_LEVEL lvl = 0;
	TRANSPORT trans = TCP;
	char * ipaddress;
	__u32 daddr;
	int mask;

	while(1) {
		c = getopt(argc, argv, "i:l:t:hq");

		if (c == -1)
			break;

		switch(c) {
		case 'i':
			sleep_interval = atoi(optarg);
			break;
		case 'l':
			i = atoi(optarg);
			if (i == 1) { 
				lvl = ONE;
			} else if (i == 5) {
				lvl = FIVE;
			} else if (i == 15) {
				lvl = FIFTEEN;
			} else {
				printf("%d is an invalid level.\nValue must be 1, 5, or 15\n", lvl);
				exit(-1);
			}

			break;
		case 'h':
			help();
			break;
		case 'q':
			quiet_mode = 1;
			break;
		case 't':
			if (strncmp(optarg, "arp", 3) == 0) {
				trans = ARP;
			} else if (strncmp(optarg, "tcp", 3) == 0) {
				trans = TCP;
			} else if (strncmp(optarg, "ip", 2) == 0) {
				trans = IP;
			} else {
				printf("%s is an invalid transport agent.\nValue must be arp, tcp, or ip\n", optarg);
				exit(-1);
			}
			break;
		default:
			break;
		}	
	}

	ipaddress = (char *)malloc(strlen(argv[optind]) + 1);
	strncpy(ipaddress, argv[optind], strlen(argv[optind]));
	daddr = addr_char_to_u32(ipaddress);

//	mask = atoi(ipaddress);
	mask = 32;
	if ((mask > 32) || (mask < 0)) {
		printf("%d is an invalid mask length.\nValue must be between 0 and 32\n", mask);
		exit(-1);
	}

	while (1) {
		double avgs[3];
		int loads;

		errno = 0;
		loads = RPSgetloadavg(avgs, 3);
		if (loads == -1) {
			perror("Error Getting load average");
			exit(1);
		}
		queue_data(avgs[lvl], daddr, mask, trans);
		sleep(sleep_interval);
	}

return 0;
}


void queue_data(double avg, __u32 addr, int mask, TRANSPORT t) {
	FILE * fp;
	__u8 c;
	__u16 d;
	int i;

	fp = fopen(IPDSENDFILE, "w");
	if (fp == NULL) {
		printf("Could not open ipdsend file\n");
		exit(-1);
	}


	if (t == ARP) {
		c = 4;
		fwrite(&c, sizeof(__u8), 1, fp);
		i = (int)(avg * 100);
		d = (__u16)i;
		fwrite(&d, sizeof(__u16), 1, fp);
		c = 0;
		fwrite(&c, sizeof(__u8), 1, fp);
	} else {
		if (t == IP) {
			c = 1;
		} else if (t == TCP) {
			c = 2;
		}
		fwrite(&c, sizeof(__u8), 1, fp);
		fwrite(&addr, sizeof(__u32), 1, fp);
		c = mask;
		fwrite(&c, sizeof(__u8), 1, fp);
		c = 1;
		fwrite(&c, sizeof(__u8), 1, fp);
		c = 0;
		fwrite(&c, sizeof(__u8), 1, fp);
		i = (int)(avg * 100);
		d = (__u16)i;
		fwrite(&d, sizeof(__u16), 1, fp);
		c = 15+16;
		fwrite(&c, sizeof(__u8), 1, fp);
	}

	fclose(fp);

	avg = ((double)d / 100);
	if (!quiet_mode)
		printf("LoadAvg: %lf sent to %u.%u.%u.%u/%d\n", avg,
			(addr & 0x000000ff),
			(addr & 0x0000ff00) >> 8,
			(addr & 0x00ff0000) >> 16,
			(addr & 0xff000000) >> 24,
			mask);
}


__u32 addr_char_to_u32(char * address) {
	__u32 addr = 0;
	char * holder;

	addr += strtol(strsep(&address, "."), &holder, 10) << 24;
	addr = addr >> 8;
	addr += strtol(strsep(&address, "."), &holder, 10) << 24;
	addr = addr >> 8;
	addr += strtol(strsep(&address, "."), &holder, 10) << 24;
	addr = addr >> 8;
	addr += strtol(address, &holder, 10) << 24;

return addr;
}
