#include "ldavgc.h"

/*
 * -h
 * -i interval
 * -t transport
 * ipaddress
 *
 */

ctrl * control = NULL;
host_t * ghost_list = NULL;
gdefs_t gdefs;
int gnum_listeners = 0;
int gfiletag = 0;

void help() {
	printf("ldavgc\n");
	printf("Usage: ldavgc [-i interval] [-t transport]\n\n");
	printf("\ti interval : time interval taken between checking for new values\n");
	printf("\tt transport : which transport mechanism to use (arp|tcp|ip)\n");
	exit(0);
}


void start_win() {
	if (!(stdscr=initscr())) {
		fprintf(stderr, "inp: initscr() failed\n\n");
		exit(1);
	}

	if (has_colors() == FALSE) {
		endwin();
		fprintf(stderr, "Terminal does not support colors...\n");
		exit(1);
	}

	start_color();

	init_pair(1, COLOR_RED, COLOR_BLACK);
	init_pair(2, COLOR_BLUE, COLOR_WHITE);

	init_pair(10, COLOR_WHITE, COLOR_BLUE);
	init_pair(11, COLOR_BLACK, COLOR_GREEN);
	init_pair(12, COLOR_BLACK, COLOR_YELLOW);
	init_pair(13, COLOR_WHITE, COLOR_RED);

	init_pair(20, COLOR_BLUE, COLOR_WHITE);
	init_pair(21, COLOR_GREEN, COLOR_WHITE);
	init_pair(22, COLOR_YELLOW, COLOR_WHITE);
	init_pair(23, COLOR_RED, COLOR_WHITE);

}


int main(int argc, char ** argv) {
	int c;
	int check_interval = 10;
	char * ipaddress;
	message_type transport = TCP_URGENT;

	while(1) {
		c = getopt(argc, argv, "i:t:h");

		if (c == -1)
			break;

		switch(c) {
			case 'i':
				check_interval = atoi(optarg);
				break;
			case 't':
				if (strcmp("ip", optarg) == 0) {
					transport = IP_DFRAG;
				} else if (strcmp("arp", optarg) == 0) {
					transport = ARP;
				} else if (strcmp("tcp", optarg) == 0) {
					transport = TCP_URGENT;
				} else {
					help();
				}
				break;
			case 'h':
				help();
				break;
			default:
				break;
		}
	}

	gdefs.interval = check_interval;
	gdefs.transport = transport;

//	if (optind >= argc) {
//		help();
//	}

	start_win();

	drawwin();
return 0;
}

void draw_banner() {
        wattron(stdscr,COLOR_PAIR(2));
        mvwhline(stdscr, LINES-2, 0, ' ', COLS);
        mvwhline(stdscr, LINES-1, 0, ' ', COLS);
        mvwaddstr(stdscr, LINES-2, 0, "LoadBanner v.01 --\t[^A] Add hosts\t[^D] Delete hosts\t[q] Quit");
        doupdate();
        wattroff(stdscr,COLOR_PAIR(2));
}

void drawwin() {
	char * foo = NULL;
	int finished = 0;
	host_t * hosts = NULL, * temp;
	wchar_t ch;
	int counter = 0;
	int view_threshold;
	int host_index = 0;

	draw_banner();

	while (!finished) {
		nodelay(stdscr, TRUE);
		curs_set(0);
		noecho();

		view_threshold = (COLS / 16) * (((LINES- 2) / 3));


		if ((ch = wgetch(stdscr)) != ERR) {
			if (ch == CTRL('A')) {
				add_host(add_host_setup());
			} else if (ch == CTRL('D')) {
				delete_ctrl(del_host_setup());
				wclear(stdscr);
				draw_banner();
			} else if (ch == CTRL('N')) {
				//screen scrolling
				if (host_index + view_threshold < gnum_hosts)
					host_index += view_threshold;
				wclear(stdscr);
				draw_banner();
			} else if (ch == CTRL('P')) {
				if (host_index - view_threshold < 0)
					host_index = 0;
				else 
					host_index -= view_threshold;
				wclear(stdscr);
				draw_banner();
			} else if (ch == 'q') {
				finished = 1;
				delete_listeners();
				break;
			} else if (ch == KEY_RESIZE) {
				wclear(stdscr);
				draw_banner();
			}
		}


		if (counter >= gdefs.interval * 100) {
			update_averages();
			sort_hosts();
			counter = 0;
		}

		temp = ghost_list;
		if (temp != NULL) {
			int i = 0, j = 0;
			int host_i = 0 ;
			while (temp != NULL) {
				int age_pair = 0, load_pair = 0;

				if (host_i < host_index) {
					host_i++;
					temp = temp->next;
					continue;
				}

				age_pair = temp->age;
				if (age_pair > 3) 
					age_pair = 3;

				//wattron(stdscr, COLOR_PAIR(age_pair + 20));
				wattron(stdscr, COLOR_PAIR(2));

				mvwhline(stdscr, i, j, ' ', 15);
				mvwhline(stdscr, i+1, j, ' ', 15);

				mvwprintw(stdscr, i, j, "%u.%u.%u.%u", \
					(temp->address & 0x000000ff), \
					(temp->address & 0x0000ff00) >> 8, \
					(temp->address & 0x00ff0000) >> 16, \
					(temp->address & 0xff000000) >> 24);

				//wattroff(stdscr, COLOR_PAIR(age_pair + 20));
				wattroff(stdscr, COLOR_PAIR(2));

				load_pair = temp->loadavg/2;
				if (load_pair > 3)
					load_pair = 3;
				if (load_pair < 0)
					load_pair = 3;					

				wattron(stdscr, COLOR_PAIR(load_pair + 10));

				mvwhline(stdscr, i+1, j, ' ', 7);

				if (temp->loadavg < 0) {
					mvwaddstr(stdscr, i+1, j, "INVALID");
				} else {	
					mvwprintw(stdscr, i+1, j, " %.2lf", temp->loadavg);
				}

				wattroff(stdscr, COLOR_PAIR(load_pair + 10));

				wattron(stdscr, COLOR_PAIR(age_pair + 10));

				mvwhline(stdscr, i+1, j+8, ' ', 7);
				mvwprintw(stdscr, i+1, j+8, "Age: %d", temp->age * gdefs.interval);

				wattroff(stdscr, COLOR_PAIR(age_pair + 10));

				temp = temp->next;
				j += 16;
				if (j > COLS - 16) {
					i += 3;
					j = 0;
				}


				if (i > (LINES) - 5) {
					wattron(stdscr, COLOR_PAIR(2));
					mvwprintw(stdscr, LINES - 1, 0, "Too many hosts to display: [^N] Move Forwards");
					if (host_index > 0)
						wprintw(stdscr,  "\t[^P] Move Backwards");
					wattroff(stdscr, COLOR_PAIR(2));
					break;

				} else if (host_index > 0) {
					wattron(stdscr, COLOR_PAIR(2));
					mvwprintw(stdscr, LINES - 1, 0, "Too many hosts to display: [^P] Move Backwards");
					wattroff(stdscr, COLOR_PAIR(2));
				}


				host_i++;
			}
		}
		doupdate();

		napms(10);
		counter++;
	}
	endwin();
}


char * add_host_setup() {
	char * str = (char *)malloc(1024);
	WINDOW * ctrlwin = newwin(6, 30, LINES/3, COLS/3);
	PANEL * panel;

	panel = new_panel(ctrlwin);

	wbkgd(ctrlwin, ' '|COLOR_PAIR(2));
	box(ctrlwin, 0, 0);
	mvwaddstr(ctrlwin, 0, 1, "Add Hosts");
	mvwaddstr(ctrlwin, 1, 1, "Ex: tcp:192.168.1.100/32");
	mvwaddstr(ctrlwin, 2, 1, ":> ");
	curs_set(1);
	echo();
	show_panel(panel);
	update_panels();

	doupdate();
	wgetnstr(ctrlwin, str, 1024);
	hide_panel(panel);
	curs_set(0);
	noecho();
	update_panels();
	doupdate();

	return str;
}


int add_host(char * address) {	
	int mask = 32;
	message_type trans = gdefs.transport;
	char * filename;
	ctrl * temp = control;


	if (control == NULL) {
		control = malloc(sizeof(ctrl));
		control->next = NULL;
		control->prev = NULL;
	} else {
		control->prev = malloc(sizeof(ctrl));
		control->prev->next = control;
		control = control->prev;
		control->prev = NULL;
	}


	if (strchr(address, ':') != NULL) {
		char * temp;
		temp = strsep(&address, ":");
		if (strncmp(temp, "arp", 3) == 0) {
			trans = ARP;
		} else if (strncmp(temp, "tcp", 3) == 0) {
			trans = TCP_URGENT;
		} else if (strncmp(temp, "ip", 2) == 0) {
			trans = IP_DFRAG;
		} else {
			wattron(stdscr,COLOR_PAIR(2));
			mvwaddstr(stdscr, LINES - 1, 0, "Invalid transport tag");
			doupdate();
			wattroff(stdscr, COLOR_PAIR(2));
			control = control->next;
			free(address);
			return -1;
		}
	}

	if (strchr(address, '/') != NULL) {
		char * temp;
		temp = strsep(&address, "/");
		mask = atoi(address);
		if ((mask > 32) || (mask < 0)){
			wattron(stdscr, COLOR_PAIR(2));
			mvwaddstr(stdscr, LINES - 1, 0, "Invalid subnet mask");
			doupdate();
			wattroff(stdscr, COLOR_PAIR(2));
			control = control->next;
			free(address);
			return -1;
		}
		address = temp;
	}
	
	if (check_valid_ip(address) != 0) {
		wattron(stdscr, COLOR_PAIR(2));
		mvwaddstr(stdscr, LINES - 1, 0, "Unrecognizable address");
		doupdate();
		wattroff(stdscr, COLOR_PAIR(2));
		control = control->next;
		free(temp);
		return -1;
	}		

	filename = malloc(strlen(LDAVGC_FILENAME) + 4);
	snprintf(filename, strlen(LDAVGC_FILENAME) + 5, "%s%d", LDAVGC_FILENAME, gfiletag);

	if (ipd_add_receiver(address, mask, trans, filename, 0) != 0) {
		wattron(stdscr, COLOR_PAIR(2));
		mvwaddstr(stdscr, LINES - 1, 0, "Could not register listener");
		doupdate();
		wattroff(stdscr, COLOR_PAIR(2));
		control = control->next;
		free(temp);
		return -1;
	}

	control->netmask = mask;
	control->address = addr_char_to_u32(address);
	control->filename = filename;
	control->transport = trans;

	gfiletag++;
	gnum_listeners++;
}


int del_host_setup() {
	char * str = (char*)malloc(1024);
	WINDOW * ctrlwin = newwin(6+gnum_listeners, 30, LINES/3, COLS/3);
	PANEL * panel;
	int i = 0;
	ctrl * temp = control;

	panel = new_panel(ctrlwin);
	wbkgd(ctrlwin, ' '|COLOR_PAIR(2));
	box(ctrlwin, 0, 0);
	mvwaddstr(ctrlwin, 0, 1, "Delete Hosts");
	for (i=0; i < gnum_listeners; i++) {
		if(temp->transport == TCP_URGENT) {
			mvwprintw(ctrlwin, 1+i, 1, "%d: tcp:%u.%u.%u.%u/%d", i,
				(temp->address & 0x000000ff),
				(temp->address & 0x0000ff00) >> 8, 
				(temp->address & 0x00ff0000) >> 16,
				(temp->address & 0xff000000) >> 24,
				temp->netmask);

		} else if(temp->transport == IP_DFRAG) {
			mvwprintw(ctrlwin, 1+i, 1, "%d: ip:%u.%u.%u.%u/%d", i,
				(temp->address & 0x000000ff),
				(temp->address & 0x0000ff00) >> 8, 
				(temp->address & 0x00ff0000) >> 16,
				(temp->address & 0xff000000) >> 24,
				temp->netmask);

		}  else if(temp->transport == ARP) {
			mvwprintw(ctrlwin, 1+i, 1, "%d: arp:%u.%u.%u.%u/%d", i,
				(temp->address & 0x000000ff),
				(temp->address & 0x0000ff00) >> 8, 
				(temp->address & 0x00ff0000) >> 16,
				(temp->address & 0xff000000) >> 24,
				temp->netmask);

		} 
		temp = temp->next;
	}
	mvwaddstr(ctrlwin, 1 + gnum_listeners, 1, ":> ");

	curs_set(1);
	echo();
	show_panel(panel);
	update_panels();

	doupdate();
	wgetnstr(ctrlwin, str, 1024);
	hide_panel(panel);
	curs_set(0);
	noecho();
	update_panels();
	doupdate();
	
	if (strlen(str) == 0)
		return -1;

	return atoi(str);
}



int delete_ctrl(int ctrl_index) {
	ctrl * temp = control;
	int i = 0;

	if ((ctrl_index < 0) || (ctrl_index >= gnum_listeners)) {
		wattron(stdscr, COLOR_PAIR(2));
		mvwaddstr(stdscr, LINES - 1, 0, "Host index out of range");
		doupdate();
		wattroff(stdscr, COLOR_PAIR(2));
		return -1;
	}

	for (i = 0; i < ctrl_index; i++) {
		temp = temp->next;
	}

	if (temp == NULL)
		return -1;

	if (i == 0)
		control = temp->next;

	ipd_delete_receiver(temp->filename);

	clear_hosts(temp->address, temp->netmask);

	free(temp->filename);

	if (temp->prev != NULL)
		temp->prev->next = temp->next;
	if (temp->next != NULL)
		temp->next->prev = temp->prev;

	free(temp);
	gnum_listeners--;
	return 0;
}



void delete_listeners() {
	ctrl * temp = control;

	while (temp != NULL) {
		ipd_delete_receiver(temp->filename);
		clear_hosts(temp->address, temp->netmask);
		temp = temp->next;
		free(temp);
	}
	ghost_list = NULL;
	control = NULL;
return;
}

void clear_hosts(long address, int netmask) {
	host_t * temp = ghost_list;
	
	while (temp != NULL) {
		if (ipcmp(address, netmask, temp->address) == 0) {
			if (temp == ghost_list) {
				ghost_list = temp->next;
			}
			delete_host(temp);
		}

		temp = temp->next;
	}

	return;
}


void delete_host(host_t * host) {
	if (host == NULL) 
		return;

	if (host->next != NULL)
		host->next->prev = host->prev;
	if (host->prev != NULL)
		host->prev->next = host->next;
	free(host);
}

int update_hosts(host_t * host_list, message_t msg, message_type transport) {
	host_t * temp = host_list;

	while (temp != NULL) {
		if (transport == TCP_URGENT) {
			if (msg.tcp_msg->flag != 15)
				return 0;

			if (temp->address == msg.tcp_msg->saddr) {
				int avg;
				avg = msg.tcp_msg->message[1];
				avg = (avg << 8);
				avg +=  msg.tcp_msg->message[0];
				temp->loadavg = ((double)avg / 100);
				temp->age = 0;
				return 0;
			}
		} else if (transport == IP_DFRAG) {
			if (msg.ip_msg->flag != 1) 
				return 0;

			if(temp->address == msg.ip_msg->saddr) {
				int avg;
				avg = msg.ip_msg->message[1];
				avg = (avg << 8);
				avg += msg.ip_msg->message[0];
				temp->loadavg = ((double)avg / 100);
				temp->age = 0;
				return 0;
			}
		} else if (transport == ARP) {
			if (*(msg.arp_msg->message) == 0)
				return 0;

			if (temp->address == msg.arp_msg->saddr) {
				int avg;
				avg = msg.arp_msg->message[1];
				avg = (avg << 8);
				avg += msg.arp_msg->message[0];
				temp->loadavg = ((double)avg / 100);
				temp->age = 0;
				return 0;
			}
		}
		temp = temp->next;
	}
return -1;
}



void create_new_host(message_t msg, message_type transport) {
	int avg;
	host_t * temp;
	temp = (host_t*)malloc(sizeof(host_t));


	if (transport == TCP_URGENT) {
		if (msg.tcp_msg->flag != 15)
			return;

		avg = msg.tcp_msg->message[1];
		avg = (avg << 8);
		avg +=  msg.tcp_msg->message[0];
		temp->loadavg = ((double)avg / 100);
		temp->address = msg.tcp_msg->saddr;
		temp->age = 0;
	} else if (transport == IP_DFRAG) {
		if (msg.ip_msg->flag != 1) 
			return;

		avg = msg.ip_msg->message[1];
		avg = (avg << 8);
		avg += msg.ip_msg->message[0];
		temp->loadavg = ((double)avg / 100);
		temp->address = msg.ip_msg->saddr;
		temp->age = 0;
	} else if (transport == ARP) {
		if (*(msg.arp_msg->message) == 0)
			return;

		avg = msg.arp_msg->message[1];
		avg = (avg << 8);
		avg += msg.arp_msg->message[0];
		temp->loadavg = ((double)avg / 100);
		temp->address = msg.tcp_msg->saddr;
		temp->age = 0;
	}

/*	temp->address = msg.tcp_msg->saddr;
	avg = msg.tcp_msg->message[1];
	avg = (avg << 8) + msg.tcp_msg->message[0];
	temp->loadavg = ((double)avg / 100);
	temp->age = 0;
*/
	if (ghost_list == NULL) {
		ghost_list = temp;
		temp->next = NULL;
		temp->prev = NULL;
	} else {
		ghost_list->prev = temp;
		temp->next = ghost_list;
		temp->prev = NULL;
		ghost_list = temp;
	}
	gnum_hosts++;
	return;
}


void age_hosts(host_t * host_list) {
	host_t * temp = host_list;

	while(temp != NULL) {
		temp->age++;
		temp = temp->next;
	}
}


void sort_hosts() {
	host_t * temp = ghost_list;
	host_t * holder = temp;	
	host_t * tail = NULL, * head = NULL;
	__u32 tempaddr1, tempaddr2;


	while (temp != NULL) {
		while (temp != NULL) {
			tempaddr1 = 0;
			tempaddr1 += (temp->address & 0x000000ff);
			tempaddr1 = tempaddr1 << 8;
			tempaddr1 += (temp->address & 0x0000ff00) >> 8;
			tempaddr1 = tempaddr1 << 8;
			tempaddr1 += (temp->address & 0x00ff0000) >> 16;
			tempaddr1 = tempaddr1 << 8;
			tempaddr1 += (temp->address & 0xff000000) >> 24;

			tempaddr2 = 0;
			tempaddr2 += (holder->address & 0x000000ff);
			tempaddr2 = tempaddr2 << 8;
			tempaddr2 += (holder->address & 0x0000ff00) >> 8;
			tempaddr2 = tempaddr2 << 8;
			tempaddr2 += (holder->address & 0x00ff0000) >> 16;
			tempaddr2 = tempaddr2 << 8;
			tempaddr2 += (holder->address & 0xff000000) >> 24;
		
			if (tempaddr1 > tempaddr2)
			// if (temp->address > holder->address)
				holder = temp;
			temp = temp->next;
		}
	
		if (ghost_list->address == holder->address)
			ghost_list = holder->next;		

		if (holder->next != NULL)
			holder->next->prev = holder->prev;
		if (holder->prev != NULL)
			holder->prev->next = holder->next;	
			
		holder->next = head;
		holder->prev = NULL;
		if (head != NULL)
			head->prev = holder;
		head = holder;
		temp = ghost_list;
		holder = temp;
	}

	ghost_list = head;
return;
}

void update_averages() {
	int i = 0, j = 0;
	host_t * host = NULL, * head = NULL;
	ipd_receiver_t * msgs;
	message_queue_t * msg_q;
	ctrl * temp = control;
	age_hosts(ghost_list);
	
	
	for (j = 0; j < gnum_listeners; j++) {

		msgs = retrieve_messages(temp->filename);

		if (msgs == NULL)
			return;

	//	mvwprintw(stdscr, LINES - 1, 0, "Messages Found: %d", msgs->num_msgs);
		msg_q = msgs->msg_queue;
		for (i = 0; i < msgs->num_msgs; i++) {
			message_queue_t * temp_q;
			if (update_hosts(ghost_list, msg_q->msg, temp->transport) != 0)
				create_new_host(msg_q->msg, temp->transport);
			temp_q = msg_q;
			msg_q = msg_q->next;
			free(temp_q);
		}

		temp = temp->next;
//TAKE THIS OUT
//TESTING PURPOSES ONLY
sleep(1);
///////////////////
/////////////////
		free(msgs);
	}

	return;
}


int check_valid_ip(char * address) {
        int len = strlen(address);
        int i = 0, temp, block_count = 0;

        for (i=0; i < len; i++) {
                if ((address[i] >= 48) && (address[i] <= 57)) {
                        temp = address[i] - 48;
                } else if (address[i] == 46) {
                        block_count++;
                        temp = 0;
                        if (block_count > 3)
                                return -1;
                } else {
                        return -1;
                }

                temp *= 10;

                if (temp > 255) {
                        return -1;
                }
        }

	if (block_count < 3) 
		 return -1;
return 0;
}

__u32 addr_char_to_u32(char * address) {
        __u32 addr = 0;
        char * holder;  
        
        addr += strtol(strsep(&address, "."), &holder, 10) << 24;
        addr = addr >> 8;
        addr += strtol(strsep(&address, "."), &holder, 10) << 24;
        addr = addr >> 8;
        addr += strtol(strsep(&address, "."), &holder, 10) << 24;
        addr = addr >> 8;       
        addr += strtol(address, &holder, 10) << 24;
                
return addr;
}

int ipcmp(long address, int netmask, long test_address) {
	__u32 mask, addr, addr_test;

	mask = 0xffffffff;
	mask = mask >> (32 - netmask);
	
	addr = mask & address;
	addr_test = mask & test_address;

	if (addr == addr_test)
		return 0;

	return -1;
}
