
/*
IP Diffusion Send Module
Concept by Peter Dinda - pdinda@cs.northwestern.edu
Programming by Brian Cornell - techie@northwestern.edu
*/

#define __OPTIMIZE__

#include <linux/config.h>
#include <linux/module.h>
#include <linux/ip.h>
#include <asm/uaccess.h>
#include <asm/checksum.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/spinlock.h>
#include <linux/skbuff.h>
#include <linux/netlink.h>
#include <linux/interrupt.h>
#include <linux/sysctl.h>
#include <linux/if_ether.h>
#include <linux/in.h>
#include <linux/tcp.h>
#include <net/tcp.h>

// Structure to hold the data we're gonna send for IP and TCP
struct senddata {
	union {
		__u16 number;
		char string[2];
		struct {
			__u8 byte1;
			__u8 byte2;
		} bytes;
	} id;
	__u8	flag:1,		// Flag to send in IP packets
		reserved:4,	// Reserved bits to send in TCP packets
		ip:1,		// Whether to send this data in IP
		tcp:1,		// Whether to send this data in TCP
		unused:1;	// Extra bit, not needed at the moment
} blank_data;

//Lock to make sure only one instance at a time
static spinlock_t lock;

//Input and output proc file
static struct proc_dir_entry *proc_file;

// Structure to hold all of the data we have for IP and TCP
// Singly linked list structure
struct datalist {
	struct datalist* next;
	union {
		struct {
			__u32 daddr;
			__u8 mask;
			char repeat;
			char priority;
			struct senddata data;
		} value;
		char raw[10];
	} data;
};
static struct datalist* listhead = NULL;	// Head of the data list
int iptcpqueuesize = 0;

// Structure to hold all of the data we have for ETH
// Singly linked list structure
struct ethlist {
	struct ethlist* next;
	unsigned char daddr[6];
	__u8 mask;
	char repeat;
	char priority;
	__u16 protocol;
	char datalen;
	char data[60];
};
static struct ethlist* ethlisthead = NULL;
int ethqueuesize = 0;

// Data to be broadcast on ARP packets
char broadcast_data[60] = "";
char broadcast_datalen = 0;

// Packet interceptor written into the kernel
extern void register_xmit_packet_interceptor(void *);

// ENTRY_LEN - number of bytes put into ipdsend to queue TCP and IP data
// ETH_ENTRY_MIN - the miminum number of bytes put into ipdsend to queue ETH data
// PADDING_LEN - number of bytes pretty-print output by ipdsend for TCP and IP data
// ETH_PADDING_MIN the mimimum number of bytes pretty-print output by ipdsend for ETH data
// ETH_PADDING_MAX  "  maximum   "
// GET_IP - value that tells get_next_data to find IP data
// GET_TCP - value that tells get_next_data to find TCP data
// SEND_IP - mask on input for data to be sent via IP
// SEND_TCP - mask on input for data to be sent via TCP
// SEND_ARP - mask on input for data to be sent via ARP
// SEND_ETH - mask on input for data to be sent via ETH
// CLEAR - mask on input says to clear the specified protocol(s) rather than add something to them
#define ENTRY_LEN 11
#define ETH_ENTRY_MIN 13
#define PADDING_LEN 41
#define ETH_PADDING_MIN 40
#define ETH_PADDING_MAX 100
#define GET_IP 1
#define GET_TCP 2
#define SEND_IP 1
#define SEND_TCP 2
#define SEND_ARP 4
#define SEND_ETH 8
#define CLEAR 128

// FIX_ADDR - reverse the bytes of an address so that it can be manipulated
// FIX_ETH_ADDR - converts an ehternet (MAC) address into a nmbr than can be manipulated
// MASK_ADDR - return the address with only the bits significant given the mask
// MASK_ETH_ADDR - return a number from a MAC address that is masked properly
// REVERSE - switch the bytes in a word around
#define FIX_ADDR(addr) (__u32)(16777216 * ((unsigned char*)&addr)[0] \
                                + 65536 * ((unsigned char*)&addr)[1] \
                                  + 256 * ((unsigned char*)&addr)[2] \
			                + ((unsigned char*)&addr)[3])
#define FIX_ETH_ADDR(addr) (__u64)(1099511627776 * addr[0] \
                                    + 4294967296 * addr[1] \
				      + 16777216 * addr[2] \
				         + 65536 * addr[3] \
					   + 256 * addr[4] \
					   	 + addr[5])
#define MASK_ADDR(ADDR, MASK) (FIX_ADDR(ADDR) & ((__u32)(((__u64)1 << 32) - 1) - ((__u32)(((__u64)1 << (32 - MASK)) - 1))))
#define MASK_ETH_ADDR(ADDR, MASK) (FIX_ETH_ADDR(ADDR) & ((((__u64)1 << 48) - 1) - (((__u64)1 << (48 - MASK)) - 1)))
#define REVERSE(word) ((word * 256 + word / 256) % 65536)

// Search through all the data and based on the destination IP address, pick the best thing to send
#ifndef DEBUG
static
#endif
struct senddata* get_next_data(__u32 daddr, __u8 protocol) {
	struct datalist* curhead = listhead;
	struct datalist* prevhead = NULL;
	struct datalist* lastmatching = NULL;
	char maxmask = -1;
	char maxpriority = -128;
	char minrepeat = -1;

	// Search through the data and find the largest netmask, the highest priority, and the lowest repeat
	while (curhead != NULL) {
		// If we run into a repeat of 0, remove the data from the list
		if (curhead->data.value.repeat == 0) {
			if (prevhead == NULL) {
				listhead = curhead->next;
				kfree(curhead);
				curhead = listhead;
				iptcpqueuesize--;
			} else {
				prevhead->next = curhead->next;
				kfree(curhead);
				curhead = prevhead->next;
				iptcpqueuesize--;
			}
		} else {
			if ((MASK_ADDR(curhead->data.value.daddr, curhead->data.value.mask) ==
			     MASK_ADDR(daddr,                     curhead->data.value.mask)) &&
			    (((protocol == GET_IP) && (curhead->data.value.data.ip == 1)) ||
			     ((protocol == GET_TCP) && (curhead->data.value.data.tcp == 1)))) {
				if (curhead->data.value.mask == maxmask) {
					if (curhead->data.value.priority == maxpriority) {
						if ((minrepeat == -1) ||
						((curhead->data.value.repeat <= minrepeat) &&
						(curhead->data.value.repeat > 0))) {
							minrepeat = curhead->data.value.repeat;
							lastmatching = curhead;
						}
					} else if (curhead->data.value.priority > maxpriority) {
						maxpriority = curhead->data.value.priority;
						minrepeat = curhead->data.value.repeat;
						lastmatching = curhead;
					}
				} else if (curhead->data.value.mask > maxmask) {
					maxmask = curhead->data.value.mask;
					maxpriority = curhead->data.value.priority;
					minrepeat = curhead->data.value.repeat;
					lastmatching = curhead;
				}
			}
			prevhead = curhead;
			curhead = curhead->next;
		}
	}

	// No matching data found, send nothing
	if (maxmask == -1) {
		return &blank_data;
	}

	if (lastmatching->data.value.repeat > 0)
		lastmatching->data.value.repeat--;
	return &lastmatching->data.value.data;
}

// Same thing as above, but looking for ETH data
struct ethlist* get_next_eth_data(char daddr[6], __u16 protocol) {
	struct ethlist* curhead = ethlisthead;
	struct ethlist* prevhead = NULL;
	struct ethlist* lastmatching = NULL;
	char maxmask = -1;
	char maxpriority = -128;
	char minrepeat = -1;

	// Search through the data and find the largest netmask, highest priority, and lowest repeat
	while (curhead != NULL) {
		// If we run into a repeat of 0, remove the data from the list
		if (curhead->repeat == 0) {
			if (prevhead == NULL) {
				ethlisthead = curhead->next;
				kfree(curhead);
				curhead = ethlisthead;
				ethqueuesize--;
			} else {
				prevhead->next = curhead->next;
				kfree(curhead);
				curhead = prevhead->next;
				ethqueuesize--;
			}
		} else {
			if ((MASK_ETH_ADDR(curhead->daddr, curhead->mask) ==
			     MASK_ETH_ADDR(daddr,          curhead->mask)) &&
			    (curhead->protocol == protocol)) {
				if (curhead->mask == maxmask) {
					if (curhead->priority == maxpriority) {
						if ((minrepeat == -1) ||
						((curhead->repeat <= minrepeat) &&
						(curhead->repeat > 0))) {
							minrepeat = curhead->repeat;
							lastmatching = curhead;
						}
					} else if (curhead->priority > maxpriority) {
						maxpriority = curhead->priority;
						minrepeat = curhead->repeat;
						lastmatching = curhead;
					}
				} else if (curhead->mask > maxmask) {
					maxmask = curhead->mask;
					maxpriority = curhead->priority;
					minrepeat = curhead->repeat;
					lastmatching = curhead;
				}
			}
			prevhead = curhead;
			curhead = curhead->next;
		}
	}

	// No matching data found, send nothing
	if (maxmask == -1) {
		return NULL;
	}

	if (lastmatching->repeat > 0)
		lastmatching->repeat--;
	return lastmatching;
}

// Internal handler for IP packets
#ifndef DEBUG
static
#endif
void inet_packet_handler(struct sk_buff *skb)
{
	struct senddata* data;

	// For debugging purposes, print the address
#ifdef DEBUG
	printk("Dest addr: %u.%u.%u.%u\n", NIPQUAD(skb->nh.iph->daddr));
#endif

	// Check the Don't Fragment flag
	if (skb->nh.iph->frag_off & (__u16)64) {
		data = get_next_data(skb->nh.iph->daddr, GET_IP);

		// reverse the data and put it in the packet
		skb->nh.iph->id = REVERSE(data->id.number);
		skb->nh.iph->frag_off = (skb->nh.iph->frag_off & (__u16)65407) | (128 * data->flag);
		skb->nh.iph->check = 0;

		// rechecksum
		skb->nh.iph->check = ip_fast_csum((unsigned char *)skb->nh.iph, skb->nh.iph->ihl);

		// for debugging, print packet info
#ifdef DEBUG
		printk("DF = 1, ID = %04x, flag = %d, check = %04x\n", data->id.number, data->flag, skb->nh.iph->check);
#endif
	} else {
#ifdef DEBUG
		printk("DF = 0, skipped\n");
#endif
	}

}

// Internal handler for TCP packets
#ifndef DEBUG
static
#endif
void tcp_packet_handler(struct sk_buff *skb)
{
	struct senddata* data;

#ifdef DEBUG
		printk("Begin TCP handler...\n");
#endif
	// Check to make sure it's not urgent
	if (skb->h.th->urg == 0) {
#ifdef DEBUG
		printk("Get data...\n");
#endif
		data = get_next_data(skb->nh.iph->daddr, GET_TCP);

		// reverse the data and put it in the packet
#ifdef DEBUG
		printk("Set data...\n");
#endif
		skb->h.th->urg_ptr = REVERSE(data->id.number);
		skb->h.th->res1 = data->reserved;

		// rechecksum
#ifdef DEBUG
		printk("Rechecksum...\n");
		printk("Len: %d, ihl: %d, ethhdr: %d\n", skb->len, skb->nh.iph->ihl, sizeof(struct ethhdr));
		printk("Len: %d or %d\n", skb->len - ((int)(skb->h.th) - (int)(skb->data)), skb->len - skb->nh.iph->ihl * 4);
		if ((skb->len - ((int)(skb->h.th) - (int)(skb->data))) != skb->len - skb->nh.iph->ihl * 4 - sizeof(struct ethhdr)) {
			printk("Different length!\n");
		}
#endif
		skb->h.th->check = 0;
		skb->h.th->check = csum_tcpudp_magic(
			skb->nh.iph->saddr, skb->nh.iph->daddr,
			skb->len - skb->nh.iph->ihl * 4 - sizeof(struct ethhdr), IPPROTO_TCP,
			csum_partial((char *)(skb->h.th), skb->len - skb->nh.iph->ihl * 4 - sizeof(struct ethhdr), 0));

#ifdef DEBUG
		printk("URG = 0, URG_PTR = %04x, flag = %d, check = %04x\n", data->id.number, data->flag, skb->h.th->check);
#endif
	} else {
#ifdef DEBUG
		printk("URG = 1, skipped\n");
#endif
	}

}

// Internal handler for ARP packets
#ifndef DEBUG
static
#endif
void arp_packet_handler(struct sk_buff *skb) {
	char* freedata;
	int freelen;

	// Find the end of the packet and how much room is left
	freedata = (char *)(skb->data + skb->len);
	freelen = 60 - skb->len;
	if (broadcast_datalen < freelen)
		freelen = broadcast_datalen;

	// Put broadcast data there if there's room
	if (freelen > 0) {
		strncpy(freedata, broadcast_data, freelen);
		skb->len += freelen;
	}

}

// Internal handler for ETH packets
#ifndef DEBUG
static
#endif
void eth_packet_handler(struct sk_buff *skb) {
	char* freedata;
	int freelen;
	struct ethlist* nextdata;

	// Find out if there's room, where, and how much
	freedata = (char *)(skb->data + skb->len);
	freelen = 60 - skb->len;

	// If there is any, find some data to put in it
	if (freelen > 0) {
		nextdata = get_next_eth_data(skb->mac.ethernet->h_dest, skb->protocol);
		// If there's any data, put it in the packet
		if (nextdata) {
			if (nextdata->datalen < freelen)
				freelen = nextdata->datalen;
			if (freelen > 0) {
				memcpy(freedata, nextdata->data, freelen);
				skb->len += freelen;
			}
		}
	}

}

// Handler that gets registered and called by the kernel hack
#ifndef DEBUG
static
#endif
void xmit_packet_handler(struct sk_buff *skb) {
	int isIP;
	int isTCP;
	int isARP;

	MOD_INC_USE_COUNT;
	local_irq_disable();
	spin_lock_bh(&lock);

#ifdef DEBUG
	printk("Test protocols...\n");
	printk("skb %08x Data %08x, iph %08x...\n", skb, skb->data, skb->nh.iph);
#endif
	isIP = (REVERSE(skb->protocol) == ETH_P_IP)
		? ((isTCP = (skb->nh.iph->protocol == IPPROTO_TCP)) * 0 + 1)
		: (isTCP = 0);
	isARP = (REVERSE(skb->protocol) == ETH_P_ARP);

#ifdef DEBUG
	printk("Test for IP...\n");
#endif
	// Pass on to IP and TCP handlers if appropriate
	if (isIP) {
#ifdef DEBUG
		printk("IP handler...\n");
#endif
		inet_packet_handler(skb);
	}
#ifdef DEBUG
		printk("Test for TCP...\n");
#endif
	if (isTCP) {
#ifdef DEBUG
		printk("TCP handler...\n");
#endif
		tcp_packet_handler(skb);
	}
	// Pass on to ARP handler if appropriate
#ifdef DEBUG
	printk("Test for ARP...\n");
#endif
	if (isARP)
		arp_packet_handler(skb);
	// Let the ETH handler see if it can do anything
#ifdef DEBUG
	printk("Send to ETH handler...\n");
#endif
	eth_packet_handler(skb);
#ifdef DEBUG
	printk("Done, unlock, etc\n");
#endif

	spin_unlock_bh(&lock);
	local_irq_enable();
	MOD_DEC_USE_COUNT;
}

// when someone reads our proc file, this gets run
#ifndef DEBUG
static
#endif
int proc_read_file(char* page, char** start, off_t off, int count, int* eof, void* data) {
	int len = 0;
	char temp[ETH_PADDING_MAX + 1];
	struct datalist* curhead = listhead;
	struct ethlist* curethhead = ethlisthead;
	int templen;

	MOD_INC_USE_COUNT;
	spin_lock_bh(&lock);

	len -= off;
//	page += off;
	*start = page;

	templen = snprintf(temp, ETH_PADDING_MAX, "IP/TCP Entries: %d\tETH Entries: %d\n", iptcpqueuesize, ethqueuesize);
	len += templen;
	if (len > count) {
		templen -= (len - count);
		len = count;
	}
	if (len > 0) {
		if (len > templen) {
			strncpy(page, temp, templen);
			page += templen;
		} else {
			strncpy(page, temp + templen - len, len);
			page += len;
		}
	}
			
	// Go through and return all the data in the list in a nice fashion
	while ((count > len) && (curhead != NULL)) {
		templen = snprintf(temp, PADDING_LEN + 1, "%u.%u.%u.%u/%u", NIPQUAD(curhead->data.value.daddr), curhead->data.value.mask);
		while (templen < 18)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, PADDING_LEN + 1, "%s\t%d", temp, curhead->data.value.priority);
		while (templen < 23)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, PADDING_LEN + 1, "%s\t%02x%02x,%d,%01x", temp, curhead->data.value.data.id.bytes.byte1, curhead->data.value.data.id.bytes.byte2, curhead->data.value.data.flag, curhead->data.value.data.reserved);
		while (templen < 32)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = (curhead->data.value.repeat == -1) ? snprintf(temp, PADDING_LEN + 1, "%s\txInf", temp) : snprintf(temp, PADDING_LEN + 1, "%s\tx%d", temp, curhead->data.value.repeat);
		while (templen < 37)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, PADDING_LEN + 1, "%s\t%c%c", temp, (curhead->data.value.data.ip == 1) ? 'I' : ' ', (curhead->data.value.data.tcp == 1) ? 'T' : ' ');
		while (templen < 40)
			temp[templen++] = ' ';
		temp[templen++] = '\n';
		temp[templen] = '\0';
		len += templen;
		if (len > count) {
			templen -= (len - count);
			len = count;
		}
		if (len > 0) {
			if (len > templen) {
				strncpy(page, temp, templen);
				page += templen;
			} else {
				strncpy(page, temp + templen - len, len);
				page += len;
			}
		}

		curhead = curhead->next;
	}
	while ((count > 0) && (curethhead != NULL)) {
		templen = snprintf(temp, ETH_PADDING_MIN + 1, "%02x:%02x:%02x:%02x:%02x:%02x/%u", curethhead->daddr[0], curethhead->daddr[1], curethhead->daddr[2], curethhead->daddr[3], curethhead->daddr[4], curethhead->daddr[5], curethhead->mask);
		while (templen < 20)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, ETH_PADDING_MIN + 1, "%s\t%d", temp, curethhead->priority);
		while (templen < 25)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, ETH_PADDING_MIN + 1, "%s\t%04x", temp, curethhead->protocol);
		while (templen < 30)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = (curethhead->repeat == -1) ? snprintf(temp, ETH_PADDING_MIN + 1, "%s\tInfx", temp) : snprintf(temp, ETH_PADDING_MIN + 1, "%s\t%dx", temp, curethhead->repeat);
		while (templen < 35)
			temp[templen++] = ' ';
		temp[templen] = '\0';
		templen = snprintf(temp, ETH_PADDING_MAX + 1, "%s\t%02d:", temp, curethhead->datalen);
		memcpy(&(temp[templen]), curethhead->data, curethhead->datalen);
		templen += curethhead->datalen;
		temp[templen++] = '\n';
		temp[templen] = '\0';
		len += templen;
		if (len > count) {
			templen -= (len - count);
			len = count;
		}
		if (len > 0) {
			if (len > templen) {
				memcpy(page, temp, templen);
				page += templen;
			} else {
				memcpy(page, temp + templen - len, len);
				page += len;
			}
		}

		curethhead = curethhead->next;
	}
	if (count > len) {
		templen = broadcast_datalen;
		memcpy(temp, broadcast_data, templen);
		temp[templen++] = '\n';
		temp[templen] = '\0';
		len += templen;
		if (len > count) {
			templen -= (len - count);
			len = count;
		}
		if (len > 0) {
			if (len > templen) {
				memcpy(page, temp, templen);
				page += templen;
			} else {
				memcpy(page, temp + templen - len, len);
				page += len;
			}
		}

	}

	*eof = (len == count) ? 0 : 1;

	spin_unlock_bh(&lock);
	MOD_DEC_USE_COUNT;

	return len;
}

// When someone writes to our proc file, this gets run
#ifndef DEBUG
static
#endif
int proc_write_file(struct file* file, const char* buffer, unsigned long count, void* data) {
	int len, i, bindex;
	char temp[ETH_ENTRY_MIN + 1];
	struct datalist* newhead = NULL;
	struct datalist* curhead = NULL;
	struct datalist* prevhead = NULL;
	struct ethlist* newethhead = NULL;
	struct ethlist* prevethhead = NULL;

	MOD_INC_USE_COUNT;
	spin_lock_bh(&lock);

	len = count;

	// Go through and add list elements for all the data passed in
	while (count > 0) {
		if (copy_from_user(temp, buffer, 1)) {
			spin_unlock_bh(&lock);
			MOD_DEC_USE_COUNT;
			return -EFAULT;
		}
		if (temp[0] & CLEAR) {
			prevhead = NULL;
			curhead = listhead;
			while (curhead != NULL) {
				newhead = curhead->next;
				if ((curhead->data.value.data.ip && temp[0] & SEND_IP) || (curhead->data.value.data.tcp && temp[0] & SEND_TCP)) {
					iptcpqueuesize--;
					kfree(curhead);
					if (curhead == listhead) {
						listhead = newhead;
					} else {
						prevhead->next = newhead;
					}
				} else {
					prevhead = curhead;
				}
				curhead = newhead;
			}
			if (temp[0] & SEND_ARP) {
				memset(broadcast_data, 0, 60);
				broadcast_datalen = 0;
			}
			if (temp[0] & SEND_ETH) {
				while (ethlisthead != NULL) {
					newethhead = ethlisthead->next;
					kfree(ethlisthead);
					ethlisthead = newethhead;
				}
				ethqueuesize = 0;
			}
			count--;
		} else if (temp[0] & SEND_IP || temp[0] & SEND_TCP) {
			if (count < ENTRY_LEN) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			if (copy_from_user(temp, buffer, ENTRY_LEN)) {
				spin_unlock_bh(&lock);
				MOD_DEC_USE_COUNT;
				return -EFAULT;
			}
			if (temp[5] > 32 || temp[5] < 0 || temp[6] < -1) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			buffer += ENTRY_LEN;
			count -= ENTRY_LEN;
			if (count == 1) count--;
			newhead = (struct datalist*)kmalloc(sizeof(struct datalist), GFP_ATOMIC);
			if (newhead == NULL) {
				spin_unlock_bh(&lock);
				MOD_DEC_USE_COUNT;
				return -ENOMEM;
			}
			for (i = 0; i < 4; i++)
				newhead->data.raw[i] = temp[i + 1];
			newhead->data.value.mask = temp[5];
			newhead->data.value.repeat = temp[6];
			newhead->data.value.priority = temp[7];
			newhead->data.value.data.id.string[0] = temp[8];
			newhead->data.value.data.id.string[1] = temp[9];
			newhead->data.value.data.ip = (temp[0] & SEND_IP) ? 1 : 0;
			newhead->data.value.data.tcp = (temp[0] & SEND_TCP) ? 1 : 0;
			newhead->data.value.data.reserved = temp[10] & 15;
			newhead->data.value.data.flag = (temp[10] & 16) ? 1 : 0;
			newhead->next = listhead;
			iptcpqueuesize++;
			listhead = newhead;
			newhead = newhead->next;
			prevhead = listhead;
			while (newhead != NULL) {
				if ((newhead->data.value.daddr == listhead->data.value.daddr) &&
				(newhead->data.value.mask  == listhead->data.value.mask) &&
				(newhead->data.value.data.ip <= listhead->data.value.data.ip) &&
				(newhead->data.value.data.tcp <= listhead->data.value.data.tcp) &&
				(newhead->data.value.repeat == -1) &&
				((newhead->data.value.data.id.number == listhead->data.value.data.id.number) ||
				(listhead->data.value.repeat == -1))) {
					prevhead->next = newhead->next;
					kfree(newhead);
					newhead = prevhead;
					iptcpqueuesize--;
				}
				prevhead = newhead;
				newhead = newhead->next;
			}
		} else if (temp[0] & SEND_ARP) {
			bindex = 0;
			buffer++;
			count -= 2;
			if (count < 0) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			copy_from_user(temp, buffer++, 1);
			bindex = temp[0];
			if (bindex > 60 || bindex < 0 || bindex > count) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			broadcast_datalen = bindex;
			copy_from_user(broadcast_data, buffer, bindex);
			buffer += bindex;
			count -= bindex;
		} else if (temp[0] & SEND_ETH) {
			if (count < ETH_ENTRY_MIN) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			if (copy_from_user(temp, buffer, ETH_ENTRY_MIN)) {
				spin_unlock_bh(&lock);
				MOD_DEC_USE_COUNT;
				return -EFAULT;
			}
			if (temp[7] > 48 || temp[7] < 0 || temp[8] < -1) {
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			buffer += ETH_ENTRY_MIN;
			count -= ETH_ENTRY_MIN;
			if (count == 1) count--;
			newethhead = (struct ethlist*)kmalloc(sizeof(struct ethlist), GFP_ATOMIC);
			if (newethhead == NULL) {
				spin_unlock_bh(&lock);
				MOD_DEC_USE_COUNT;
				return -ENOMEM;
			}
			for (i = 0; i < 6; i++)
				newethhead->daddr[i] = temp[i + 1];
			newethhead->mask = (__u8)temp[7];
			newethhead->repeat = temp[8];
			newethhead->priority = temp[9];
			newethhead->protocol = (__u8)temp[10] + 256 * (__u8)temp[11];
			bindex = temp[12];
			newethhead->datalen = (char)bindex;
			if (bindex > 60 || bindex < 0 || bindex > count) {
				kfree(newethhead);
				spin_unlock_bh(&lock);
				printk("Invalid data written to ipdsend\n");
				MOD_DEC_USE_COUNT;
				return -1;
			}
			copy_from_user(newethhead->data, buffer, bindex);
			buffer += bindex;
			count -= bindex;
			newethhead->next = ethlisthead;
			ethqueuesize++;
			ethlisthead = newethhead;
			newethhead = newethhead->next;
			prevethhead = ethlisthead;
			while (newethhead != NULL) {
				if ((strncmp(newethhead->daddr, ethlisthead->daddr, 6) == 0) &&
				(newethhead->mask  == ethlisthead->mask) &&
				(newethhead->repeat == -1) &&
				((strncmp(newethhead->data, ethlisthead->data, 60) == 0) ||
				(ethlisthead->repeat == -1))) {
					prevethhead->next = newethhead->next;
					kfree(newethhead);
					newethhead = prevethhead;
					ethqueuesize--;
				}
				prevethhead = newethhead;
				newethhead = newethhead->next;
			}
		} else {
			spin_unlock_bh(&lock);
			printk("Invalid data written to ipdsend\n");
			MOD_DEC_USE_COUNT;
			return -1;
		}
	}

	spin_unlock_bh(&lock);
	MOD_DEC_USE_COUNT;

	return len;
}

// module initialization, register handler and proc file
int init_module(void)
{
	blank_data.id.number = 0;
	blank_data.flag = 0;
	blank_data.reserved = 0;
	blank_data.ip = blank_data.tcp = 1;

	// Create /proc entry
	proc_file = create_proc_entry("ipdsend", 0644, NULL);
	if (proc_file == NULL) {
		return -ENOMEM;
	}

	proc_file->read_proc = proc_read_file;
	proc_file->write_proc = proc_write_file;
	proc_file->owner = THIS_MODULE;

	spin_lock_init(&lock);

	/* Register hook */
	register_xmit_packet_interceptor(&xmit_packet_handler);
	printk("IP Diffusion Send module loaded successfully\n");
	return 0;
}


// module uninitialization
void
cleanup_module(void)
{
	struct datalist* tempptr;
	struct ethlist* tempptr2;
	register_xmit_packet_interceptor(NULL);
	remove_proc_entry("ipdsend", NULL);
	while (listhead != NULL) {
		tempptr = listhead->next;
		kfree(listhead);
		listhead = tempptr;
	}
	while (ethlisthead != NULL) {
		tempptr2 = ethlisthead->next;
		kfree(ethlisthead);
		ethlisthead = tempptr2;
	}
	printk("IP Diffusion Send module unloaded\n");
}

