#include <linux/config.h>

#include <linux/module.h>
#include <linux/netfilter_ipv4.h>
#include <linux/ip.h>
#include <asm/uaccess.h>
#include <asm/checksum.h>
#include <linux/if_ether.h>
#include <linux/byteorder/generic.h>

#include <linux/skbuff.h>
#include <linux/proc_fs.h>
#include <linux/time.h>
#include <linux/spinlock.h>
#include <linux/netlink.h>
#include <linux/interrupt.h>
#include <linux/sysctl.h>
#include <linux/tcp.h>
#include <linux/in.h>


#define BOOL int
#define TRUE 1
#define FALSE 0

#define MESSAGE_QUEUE_LENGTH 100

#define BUF_LEN 100

#define PROC_CTRL_FILE_NAME "ipd_ctrl"
#define PROC_COLLECTION_DIR_NAME "ipd"

#define ARP_PACKET_LENGTH 47


typedef void (*pkt_handler_fn)(struct sk_buff *);
extern void register_recv_packet_interceptor(pkt_handler_fn handler);

#ifndef DEBUG
static
#endif
void pkt_router(struct sk_buff * skb);

typedef struct ip_message_t {
	__u32 src_addr;
	__u32 dst_addr;
	__u16 src_port;
	__u16 dst_port;

	int flag;
	char message[2];
	struct timeval timestamp;
} ip_message_t;

typedef struct arp_message_t {
	__u32 src_addr;
	__u32 dst_addr;

	char message[60];
	struct timeval timestamp;
} arp_message_t;

typedef struct message_t {
	union {
		arp_message_t * arpmsg;
		ip_message_t * ipmsg;
	} msg;
	struct message_t * next;
} message_t;

typedef enum message_type {ARP, IP_DFRAG, TCP_UNIQUE} message_type;
typedef enum FORMAT {HUMAN, MACHINE} FORMAT;

typedef struct ctrl_t {
	__u32 saddr;
	int subnet_mask;
	char * filename;
	BOOL blank_on_read;
	int read_count;
	FORMAT out_format;
	message_type msg_type;
	struct proc_dir_entry * proc_file;
	int msg_head_i;
	int msg_tail_i;
	int queue_size;
	int num_msgs;
	struct ctrl_t * next;
	struct ctrl_t * prev;
	message_t msg_queue[MESSAGE_QUEUE_LENGTH];
} ctrl_t;


spinlock_t proc_lock;
spinlock_t packet_lock;
int gbusy;


static int gnum_listeners;

struct proc_dir_entry * proc_collection_dir;


ctrl_t * ctrl_head = NULL;

#ifndef DEBUG
static
#endif
void dequeue_message(ctrl_t * ctrl) {
	ctrl->msg_head_i++;
	ctrl->num_msgs--;
}

#ifndef DEBUG
static
#endif
void delete_message_queue(ctrl_t * ctrl) {
	ctrl->msg_head_i = 0;
	ctrl->msg_tail_i = 0;
	ctrl->num_msgs = 0;
}

#ifndef DEBUG
static
#endif
int write_proc(struct file * file,  const char* buffer, unsigned long count, void* data) {
	ctrl_t * node;

	spin_lock_bh(&packet_lock);
	local_irq_disable();
	gbusy = 1;
	MOD_INC_USE_COUNT;
	node = data;

	delete_message_queue(node);

	MOD_DEC_USE_COUNT;
	gbusy = 0;
	local_irq_enable();
	spin_unlock_bh(&packet_lock);

	return count;
}

#ifndef DEBUG
static
#endif
int read_proc( char * page, char ** start, off_t off, int count, int * eof, void * data) {
	int len = 0;
	off_t begin = 0;
	int msg_num, msg_i;
	ctrl_t * node;


	spin_lock_bh(&packet_lock);
	local_irq_disable();
	MOD_INC_USE_COUNT;

	msg_num = 0;
	node = data;
	msg_i = node->msg_head_i;

	if (node->out_format == MACHINE) {
		if (node->msg_type == IP_DFRAG) {
			memset(page + len, IP_DFRAG, sizeof(char));
		} else if (node->msg_type == ARP) {
			memset(page + len, ARP, sizeof(char));
		} else if (node->msg_type == TCP_UNIQUE) {
			memset(page + len, TCP_UNIQUE, sizeof(char));
		}
		len += sizeof(char);
		memcpy(page + len, &(node->num_msgs), sizeof(int));
		len += sizeof(int);
	} else if (node->out_format == HUMAN) {

		if (node->msg_type == IP_DFRAG) {
			len += sprintf(page+len, "Messages: %d\tTransport: IP_DONT_FRAGMENT\n", node->num_msgs);
		} else if (node->msg_type == ARP) {
			len += sprintf(page+len, "Messages: %d\tTransport: ARP_PADDING\n", node->num_msgs);
		} else if (node->msg_type == TCP_UNIQUE) {
			len += sprintf(page+len, "Messages: %d\tTransport: TCP_URGENT\n", node->num_msgs);
		}
	}
	
	if (len+begin < off) {
		begin += len;
		len = 0;
	}

	while(msg_num < node->num_msgs) {
		if (msg_i >= MESSAGE_QUEUE_LENGTH -1 ) {
			msg_i = 0;
		}
		if (node->msg_type == IP_DFRAG) {
			if (node->out_format == HUMAN) {
				len += sprintf(page+len, "%d - %ld.%ld - %u.%u.%u.%u - %u.%u.%u.%u - %c%c%d\n", \
					msg_num, \
					node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_sec, \
					node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_usec, \
					NIPQUAD(node->msg_queue[msg_i].msg.ipmsg->src_addr), \
					NIPQUAD(node->msg_queue[msg_i].msg.ipmsg->dst_addr), \
					node->msg_queue[msg_i].msg.ipmsg->message[0], \
					node->msg_queue[msg_i].msg.ipmsg->message[1], \
					node->msg_queue[msg_i].msg.ipmsg->flag);
			} else if (node->out_format == MACHINE) {
				memcpy(page + len, &msg_num, sizeof(int));
				len += sizeof(int);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_sec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_usec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->src_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->dst_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->message[0]), sizeof(char));
				len += sizeof(char);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->message[1]), sizeof(char));
				len += sizeof(char);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->flag), sizeof(int));
				len += sizeof(int);
			}
		} else if (node->msg_type == ARP) {
			if (node->out_format == HUMAN) {
				len += sprintf(page+len,"%d - %ld.%ld - %u.%u.%u.%u - %u.%u.%u.%u - %s\n", \
					msg_num, \
					node->msg_queue[msg_i].msg.arpmsg->timestamp.tv_sec, \
					node->msg_queue[msg_i].msg.arpmsg->timestamp.tv_usec, \
					NIPQUAD(node->msg_queue[msg_i].msg.arpmsg->src_addr), \
					NIPQUAD(node->msg_queue[msg_i].msg.arpmsg->dst_addr), \
					node->msg_queue[msg_i].msg.arpmsg->message);
			} else if (node->out_format == MACHINE) {
				memcpy(page + len, &msg_num, sizeof(int));
				len += sizeof(int);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.arpmsg->timestamp.tv_sec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.arpmsg->timestamp.tv_usec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.arpmsg->src_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.arpmsg->dst_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.arpmsg->message), sizeof(char) * 60);
				len += sizeof(char) * 60;
			}
		} else if (node->msg_type == TCP_UNIQUE) {
			if (node->out_format == HUMAN) {
				len += sprintf(page+len,"%d - %ld.%ld - %u.%u.%u.%u:%d - %u.%u.%u.%u:%d - %c%c%d\n", \
					msg_num, \
					node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_sec, \
					node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_usec, \
					NIPQUAD(node->msg_queue[msg_i].msg.ipmsg->src_addr), \
					node->msg_queue[msg_i].msg.ipmsg->src_port, \
					NIPQUAD(node->msg_queue[msg_i].msg.ipmsg->dst_addr), \
					node->msg_queue[msg_i].msg.ipmsg->dst_port, \
					node->msg_queue[msg_i].msg.ipmsg->message[0], \
					node->msg_queue[msg_i].msg.ipmsg->message[1], \
					node->msg_queue[msg_i].msg.ipmsg->flag);

			} else if (node->out_format == MACHINE) {
				memcpy(page + len, &msg_num, sizeof(int));
				len += sizeof(int);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_sec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->timestamp.tv_usec), sizeof(long));
				len += sizeof(long);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->src_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->src_port), sizeof(__u16));
				len += sizeof(__u16);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->dst_addr), sizeof(__u32));
				len += sizeof(__u32);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->dst_port), sizeof(__u16));
				len += sizeof(__u16);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->message[0]), sizeof(char));
				len += sizeof(char);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->message[1]), sizeof(char));
				len += sizeof(char);
				memcpy(page + len, &(node->msg_queue[msg_i].msg.ipmsg->flag), sizeof(int));
				len += sizeof(int);
			}
		}

		if (len + begin > off+count)
			break;
		if (len+begin < off) {
			begin += len;
			len = 0;
		}
		msg_i++;
		msg_num++;
	}

 	if (msg_num == node->num_msgs) {
		*eof = 1;
	}

	if (off >= len + begin) {
		if (node->blank_on_read)
			delete_message_queue(node);

		MOD_DEC_USE_COUNT;
		local_irq_enable();
		spin_unlock_bh(&packet_lock);
		return 0;
	}
	*start = page + (off - begin);
	MOD_DEC_USE_COUNT;
	local_irq_enable();
	spin_unlock_bh(&packet_lock);

	return ((count < begin + len - off) ? count : begin + len - off);
}

#ifndef DEBUG
static
#endif
int read_ctrl_proc(char * page, char ** start, off_t off, int count, int * eof, void * data) {
	int len = 0;
	off_t begin = 0;
	ctrl_t * temp_ctrl = NULL;

	spin_lock_bh(&proc_lock);
	local_irq_disable();
	MOD_INC_USE_COUNT;
	len += sprintf(page+len,"Listeners: %d\nIP\t\t\tPROTO\t\tFILENAME\tBLANK_ON_READ\tFORMAT\n",gnum_listeners);
	if (len + begin < off) {
		begin += len;
		len = 0;
	}

	temp_ctrl = ctrl_head;
	while (temp_ctrl != NULL) {
		if (temp_ctrl->msg_type == ARP) {
			len += sprintf(page + len,"%u.%u.%u.%u/%d\tARP\t\t%s\t\t%d\t\t%c\n", \
				NIPQUAD(temp_ctrl->saddr), \
				temp_ctrl->subnet_mask, \
				temp_ctrl->filename, \
				temp_ctrl->read_count, \
				(temp_ctrl->out_format == HUMAN) ? 'H' : 'M');
		} else if (temp_ctrl->msg_type == IP_DFRAG) {
			len += sprintf(page + len,"%u.%u.%u.%u/%d\tIP_DFRAG\t%s\t\t%d\t\t%c\n", \
				NIPQUAD(temp_ctrl->saddr), \
				temp_ctrl->subnet_mask, \
				temp_ctrl->filename, \
				temp_ctrl->read_count, \
				(temp_ctrl->out_format == HUMAN) ? 'H' : 'M');
		} else if (temp_ctrl->msg_type == TCP_UNIQUE) {
			len += sprintf(page + len,"%u.%u.%u.%u/%d\tTCP_URGENT\t%s\t\t%d\t\t%c\n", \
				NIPQUAD(temp_ctrl->saddr), \
				temp_ctrl->subnet_mask, \
				temp_ctrl->filename, \
				temp_ctrl->read_count, \
				(temp_ctrl->out_format == HUMAN) ? 'H' : 'M');
		}
		if (len + begin > off + count)
			break;
		if (len + begin < off) {
			begin += len;
			len = 0;
		}
		temp_ctrl = temp_ctrl->next;
	}

	if (temp_ctrl == NULL) {
		*eof = 1;
	}

	if (off >= len + begin) {
		MOD_DEC_USE_COUNT;
		local_irq_enable();
		spin_unlock_bh(&proc_lock);
		return 0;
	}
	*start = page + (off - begin);
	MOD_DEC_USE_COUNT;
	local_irq_enable();
	spin_unlock_bh(&proc_lock);
	return ((count < begin + len - off) ? count : begin + len - off);
}


#ifndef DEBUG
static
#endif
int create_message_queue(ctrl_t * ctrl) {
	int i = 0;

	for(i = 0; i < MESSAGE_QUEUE_LENGTH; i++) {
		if (ctrl->msg_type == ARP) {
			ctrl->msg_queue[i].msg.arpmsg = kmalloc(sizeof(arp_message_t), GFP_ATOMIC);
		} else if ((ctrl->msg_type == TCP_UNIQUE) || (ctrl->msg_type == IP_DFRAG)) {
			ctrl->msg_queue[i].msg.ipmsg = kmalloc(sizeof(ip_message_t), GFP_ATOMIC);
		}
	}

	ctrl->msg_head_i = 0;
	ctrl->msg_tail_i = 0;
	
	return 0;
}

#ifndef DEBUG
static
#endif
int check_for_duplicates(char * filename) {
	ctrl_t * temp;

	spin_lock_bh(&packet_lock);
	temp = ctrl_head;
	while (temp != NULL) {
		if (strcmp(temp->filename, filename) == 0) {
			spin_unlock_bh(&packet_lock);
			return 1;
		}
		temp = temp->next;
	}
	spin_unlock_bh(&packet_lock);

return 0;
}

#ifndef DEBUG
static
#endif
int check_valid_ip(char * address) {
	int len = strlen(address);
	int i = 0, temp, block_count = 0;

	for (i=0; i < len; i++) {
		if ((address[i] >= 48) && (address[i] <= 57)) {
			temp = address[i] - 48;
		} else if (address[i] == 46) {
			block_count++;
			temp = 0;
			if (block_count > 3)
				return -1;
		} else {
			return -1;
		}

		temp *= 10;

		if (temp > 255) {
			return -1;
		}
	}

if (block_count < 3) 
	return -1;
return 0;
}


#ifndef DEBUG
static
#endif
int add_listener(char * conf_line) {
	char  *temp_str, * holder, * address;
	int str_len = 0;
	ctrl_t * node;
	int temp_len;


	if (conf_line == NULL) {
		printk("Error in IPDRecv: malformed control line\n");
		return -1;
	}

	node = kmalloc(sizeof(ctrl_t), GFP_ATOMIC);
	if (node == NULL) {
		printk("Error: Kernel out of Memory\n");
		return -1;
	}

	str_len = strlen(conf_line);	

	temp_str = strsep(&conf_line," ");
	if (strlen(temp_str) > 18) {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node);
		return -1;
	}

	address = kmalloc(15, GFP_ATOMIC);
	snprintf(address, 15, strsep(&temp_str, "/"));

	if (check_valid_ip(address) != 0) {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(address);
		kfree(node);
		return -1;
	}

	node->saddr = 0;
	node->saddr += simple_strtol(strsep(&address, "."), &holder, 10) << 24;
	node->saddr = node->saddr >> 8;
	node->saddr += simple_strtol(strsep(&address, "."), &holder, 10) << 24;
	node->saddr = node->saddr >> 8;
	node->saddr += simple_strtol(strsep(&address, "."), &holder, 10) << 24;
	node->saddr = node->saddr >> 8;
	node->saddr += simple_strtol(address, &holder, 10) << 24;

	kfree(address);
	if (temp_str == NULL) {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node);
		return -1;
	}


	node->subnet_mask = simple_strtol(temp_str,&holder ,10);

	temp_str = strsep(&conf_line, " ");

	if (temp_str == NULL) {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node);
		return -1;
	}
		
	if (strncmp(temp_str, "ARP", 3) == 0) {
		node->msg_type = ARP;
	} else if (strncmp(temp_str, "IP_DFRAG", 7) == 0) {
		node->msg_type = IP_DFRAG;
	} else if (strncmp(temp_str, "TCP_URGENT",10) == 0) {
		node->msg_type = TCP_UNIQUE;
	} else {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node);
		return -1;
	}

	/* strsep() is supposed to return a char * to the string that occurs
	 * before the deliminator. However whenever I tried to allocate
	 * memory after a call to strsep, the returned char * got narked.
	 */
	if (conf_line == NULL) {
		printk("Error in IPDRecv: Malformed control line\n");
		kfree(node);
		return -1;
	}
	temp_str = strchr(conf_line, ' ');
	temp_len = temp_str - conf_line;
	node->filename = kmalloc(temp_len+2, GFP_ATOMIC);
	if (node->filename == NULL) {
		printk("Error: Kernel out of Memory\n");
		kfree(node);
		return -1;
	}

	memset(node->filename,0,temp_len+2);
	temp_str = strsep(&conf_line," ");
	if (temp_str == NULL) {
		printk("Error in IPDRecv: Malformed control line\n");
		kfree(node->filename);
		kfree(node);
		return -1;
	}
	memcpy(node->filename,temp_str, temp_len);

	if (check_for_duplicates(node->filename) == 1) {
		printk("Error in IpdRecv: Listener already defined with filename '%s'\n", node->filename);
		kfree(node->filename);
		kfree(node);
		return -1;
	}

	node->blank_on_read = FALSE;
	temp_str = strsep(&conf_line, " ");
	if (temp_str == NULL) {
                printk("Error in IPDRecv: malformed control line\n");
                kfree(node->filename);
                kfree(node);
                return -1;
	} else if (temp_str[0] == 'Y') {
		node->blank_on_read = TRUE;
	} else if (temp_str[0] == 'N') {
		node->blank_on_read = FALSE;
	} else {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node->filename);
		kfree(node);
		return -1;
	}

	node->out_format = HUMAN;

	if (conf_line == NULL) {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node->filename);
		kfree(node);
		return -1;
	} else if (conf_line[0] == 'H') {
		node->out_format = HUMAN;
	} else if (conf_line[0] == 'M') {
		node->out_format = MACHINE;
	} else {
		printk("Error in IPDRecv: malformed control line\n");
		kfree(node->filename);
		kfree(node);
		return -1;
	}

	kfree(conf_line);
	node->proc_file = create_proc_entry(node->filename,0644,proc_collection_dir);
	node->proc_file->read_proc = read_proc;
	node->proc_file->write_proc = write_proc;
	node->proc_file->data = node;
	create_message_queue(node);

	spin_lock_bh(&packet_lock);
	local_irq_disable();
	node->next = ctrl_head;
	if (node->next != NULL)
		node->next->prev = node;
	node->prev = NULL;

	gnum_listeners++;
	ctrl_head = node;
	local_irq_enable();
	spin_unlock_bh(&packet_lock);

	printk("IPDRecv: Added Listener -- %u.%u.%u.%u %s %d\n",NIPQUAD(node->saddr), node->filename, node->blank_on_read);

	return 0;
}

#ifndef DEBUG
static
#endif
int delete_listener(ctrl_t * ctrl) {

	if (ctrl == NULL) {
		printk("Tried to delete unregistered listener\n");
		return -1;
	}

	if (ctrl->prev != NULL)
		ctrl->prev->next = ctrl->next;
	else 
		ctrl_head = ctrl->next;

	if (ctrl->next != NULL)
		ctrl->next->prev = ctrl->prev;

	if (gnum_listeners == 1)
		ctrl_head = NULL;
	gnum_listeners--;

	delete_message_queue(ctrl);

	remove_proc_entry(ctrl->filename, proc_collection_dir);

	printk("Deleted Listener: %s\n",ctrl->filename);

	kfree(ctrl->filename);
	kfree(ctrl);

	return 0;
}

#ifndef DEBUG
static
#endif
int delete_listener_byfilename(char * conf_line) {
	ctrl_t * temp_ctrl;
	temp_ctrl = ctrl_head;

	if (conf_line == NULL) {
		return -1;
	}

	spin_lock_bh(&packet_lock);
	local_irq_disable();
	while (temp_ctrl != NULL) {
		if (strncmp(temp_ctrl->filename,conf_line,strlen(temp_ctrl->filename)) == 0) {
			int ret;
			printk("IPDRecv: %s\n",temp_ctrl->filename);
			ret = delete_listener(temp_ctrl);
			local_irq_enable();
			spin_unlock_bh(&packet_lock);
			return ret;	
		}
		temp_ctrl = temp_ctrl->next;
	}
	local_irq_enable();
	spin_unlock_bh(&packet_lock);

	return -1;
}

#ifndef DEBUG
static
#endif
void delete_all_listeners() {
	ctrl_t  * temp_ctrl;

	temp_ctrl = ctrl_head;
	while (temp_ctrl != NULL) {
		ctrl_t * temp2;

		temp2 = temp_ctrl->next;
		delete_listener(temp_ctrl);
		temp_ctrl = temp2;
	}

	ctrl_head = NULL;
	gnum_listeners = 0;
}


#ifndef DEBUG
static
#endif
int write_ctrl_proc(struct file * file,  const char* buffer, unsigned long count, void* data) {
	int len = 0;
	char * scratch_buf;

	MOD_INC_USE_COUNT;

	scratch_buf = kmalloc(sizeof(char) * (count+1), GFP_ATOMIC);
	if (scratch_buf == NULL) {
		MOD_DEC_USE_COUNT;
		return -1;
	}

	printk("Write function\n");

	if(copy_from_user(scratch_buf, buffer, count+1)) {
		MOD_DEC_USE_COUNT;
		return -EFAULT;
	}

	len = count;
	scratch_buf[count] = '\0';

	printk("Parsing command\n");
	{
		char * temp_line;

		temp_line = strsep(&scratch_buf," ");
		if (*temp_line == '\0') {
			// error out here
			MOD_DEC_USE_COUNT;
			kfree(scratch_buf);
			return -1;
		}
		printk("%s\n", temp_line);
		if (strncmp(temp_line,"add",3) ==0) {
			printk("IPDRecv Adding listener\n");
			add_listener(scratch_buf);
		} else if (strncmp(temp_line,"del",3) == 0) {
			printk("IPDRecv Deleting listener\n");
			delete_listener_byfilename(scratch_buf);
			kfree(scratch_buf);
		}
	}

	MOD_DEC_USE_COUNT;
	return len;
}


struct ip_recvdata {
	union {
		__u16 number;
		char string[2];
		struct {
			__u8 bute1;
			__u8 byte2;
		} bytes;
	} id;
	__u8 flag:1, unused:7;
};


#ifndef DEBUG
static
#endif
int check_address(ctrl_t *ctrl, struct sk_buff * pskb) {
	__u32 mask, ctrl_test, skb_test;

	#ifdef DEBUG
		printk("Address: %u.%u.%u.%u\n", NIPQUAD(ctrl->saddr));
	#endif

	if (ctrl->msg_type == ARP) {
		memcpy(&skb_test, pskb->data + 14, sizeof(__u32));
	} else if ((ctrl->msg_type == IP_DFRAG) || (ctrl->msg_type == TCP_UNIQUE)) {
		skb_test = pskb->nh.iph->saddr;
	}

	mask = 0xffffffff;
	mask = mask >> (32 - ctrl->subnet_mask);

	#ifdef DEBUG
		printk("%u.%u.%u.%u\n", NIPQUAD(mask));
	#endif

	ctrl_test = mask & ctrl->saddr;
	skb_test = mask & skb_test;	

	#ifdef DEBUG
		printk("ctrl: %u.%u.%u.%u  --  skb: %u.%u.%u.%u\n", NIPQUAD(ctrl_test), NIPQUAD(skb_test));
	#endif

	if (ctrl_test == skb_test) {
		#ifdef DEBUG
			printk("%s -- MATCH\n", __FUNCTION__);
		#endif
		return 0;
	}

	return -1;
}
/*
#ifndef DEBUG
static
#endif
int get_actual_ethernet_packet_length(sk_buff * pskb) {
	if (pskb->protocol == __constant_htons(ETH_P_ARP)) {
		return ARP_PACKET_LENGTH;
	} 
  	
	else if (pskb->protocol == __constant_htons(ETH_P_IP) ) {

	}

}
*/

#ifndef DEBUG
static
#endif
void arp_rcv(struct sk_buff * pskb) {
	ctrl_t * temp_ctrl, * host_ctrl = NULL;

	temp_ctrl = ctrl_head;
	while (temp_ctrl != NULL) {

		if (temp_ctrl->msg_type != ARP) {
			temp_ctrl = temp_ctrl->next;
			continue;
		}

		if (check_address(temp_ctrl, pskb) == 0) {
			host_ctrl = temp_ctrl;

			if (host_ctrl == NULL) {
				// no listener registered for this host
				return;
			}

			#ifdef DEBUG
				printk("Length: %d; Messages: %d\n", pskb->len, host_ctrl->num_msgs);
			#endif
			if (pskb->data + ARP_PACKET_LENGTH != '\0') {

				#ifdef DEBUG
					printk("%u.%u.%u.%u\n", *(pskb->data + 24), *(pskb->data+25), *(pskb->data+26), *(pskb->data+27));
				#endif

				if (host_ctrl->num_msgs >= MESSAGE_QUEUE_LENGTH) {
					host_ctrl->num_msgs = MESSAGE_QUEUE_LENGTH;
					if (host_ctrl->msg_head_i >= MESSAGE_QUEUE_LENGTH - 1) {
						host_ctrl->msg_head_i = 0;
					} else {
						host_ctrl->msg_head_i++;
					}

					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}

			//		host_ctrl->num_msgs--;
				} else if (host_ctrl->num_msgs != 0) {
					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}
					host_ctrl->num_msgs++;
				} else {
					host_ctrl->num_msgs++;
				}

				#ifdef DEBUG
					printk("Messages -- num: %d; head: %d; tail: %d\n", host_ctrl->num_msgs, host_ctrl->msg_head_i, host_ctrl->msg_tail_i);
				#endif

				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.arpmsg->timestamp = pskb->stamp;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.arpmsg->dst_addr = *(pskb->data + 24);
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.arpmsg->src_addr = *(pskb->data + 14);
				snprintf(host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.arpmsg->message, 60 - pskb->len, "%s", pskb->data + 28);
			}
		}
		temp_ctrl = temp_ctrl->next;
	}
	return;
}


/* The work comes in here from netfilter.c. */
#ifndef DEBUG
static
#endif
void ip_diff_rcv(struct sk_buff *pskb){
	ctrl_t * temp_ctrl, * host_ctrl = NULL;
	struct ip_recvdata data;

	#ifdef DEBUG
		printk("IP receiver entered\n");
	#endif
	temp_ctrl = ctrl_head;
	while (temp_ctrl != NULL) {

		if (temp_ctrl->msg_type != IP_DFRAG) {
			temp_ctrl = temp_ctrl->next;
			continue;
		}

		if (check_address(temp_ctrl, pskb) == 0) {
			host_ctrl = temp_ctrl;

			if (host_ctrl == NULL) {
				// no listener registered for this host
				return;
			}

			if (pskb->nh.iph->frag_off & (__u16)(64)) { 
				int flag;
				
				if ((pskb->nh.iph->frag_off & (__u16)(128))) {
					flag = 1;
				} else {
					flag = 0;
				}

				#ifdef DEBUG
					printk("Source addr: %u.%u.%u.%u\n", NIPQUAD(pskb->nh.iph->saddr));
				#endif
				data.id.number = pskb->nh.iph->id;

				if (host_ctrl->num_msgs >= MESSAGE_QUEUE_LENGTH) {
					host_ctrl->num_msgs = MESSAGE_QUEUE_LENGTH;
					if (host_ctrl->msg_head_i >= MESSAGE_QUEUE_LENGTH - 1) {
						host_ctrl->msg_head_i = 0;
					} else {
						host_ctrl->msg_head_i++;
					}

					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}

	//				host_ctrl->num_msgs--;
				} else if (host_ctrl->num_msgs != 0) {
					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}
					host_ctrl->num_msgs++;
				} else {
					host_ctrl->num_msgs++;
				}

				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->timestamp = pskb->stamp;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->src_addr = pskb->nh.iph->saddr;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->dst_addr = pskb->nh.iph->daddr;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[0] = data.id.string[1];
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[1] = data.id.string[0];
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->flag = flag;

				#ifdef DEBUG
					printk("%c%c\n", host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[0], host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[1]);
				#endif
			}
		}
		temp_ctrl = temp_ctrl->next;
	}
	return;
}

#ifndef DEBUG
static
#endif
void tcp_rcv(struct sk_buff * pskb) {
	ctrl_t * temp_ctrl, * host_ctrl = NULL;
	struct tcphdr * tcph = (struct tcphdr*)(pskb->nh.iph + ((pskb->nh.iph->ihl / 4) % 4));
	struct ip_recvdata data;

	#ifdef DEBUG
		printk("TCP Receiver entered\n");
	#endif

	temp_ctrl = ctrl_head;
	while (temp_ctrl != NULL) {
		if (temp_ctrl->msg_type != TCP_UNIQUE) {
			temp_ctrl = temp_ctrl->next;
			continue;
		}

		if (check_address(temp_ctrl, pskb) == 0) {
			host_ctrl = temp_ctrl;

			if (host_ctrl == NULL) {
				return;
			}

			#ifdef DEBUG
				printk("TCP packet received, URG=%d, res1=%d, iph=%08x, th=%08x\n", tcph->urg, tcph->res1, pskb->nh.iph, tcph);
			#endif

			if (tcph->urg == 0) {
				#ifdef DEBUG
					printk("Source addr: %u.%u.%u.%u\n", NIPQUAD(pskb->nh.iph->saddr));
				#endif
				data.id.number = tcph->urg_ptr;

				if (host_ctrl->num_msgs >= MESSAGE_QUEUE_LENGTH) {
					host_ctrl->num_msgs = MESSAGE_QUEUE_LENGTH;
					if (host_ctrl->msg_head_i >= MESSAGE_QUEUE_LENGTH - 1) {
						host_ctrl->msg_head_i = 0;
					} else {
						host_ctrl->msg_head_i++;
					}

					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}

			//		host_ctrl->num_msgs--;
				} else if (host_ctrl->num_msgs != 0) {
					if (host_ctrl->msg_tail_i >= MESSAGE_QUEUE_LENGTH -1) {
						host_ctrl->msg_tail_i = 0;
					} else {
						host_ctrl->msg_tail_i++;
					}
					host_ctrl->num_msgs++;
				} else {
					host_ctrl->num_msgs++;
				}


				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->timestamp = pskb->stamp;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->src_addr = pskb->nh.iph->saddr;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->dst_addr = pskb->nh.iph->daddr;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->src_port = pskb->h.th->source;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->dst_port = pskb->h.th->dest;
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[0] = data.id.string[1];
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[1] = data.id.string[0];
				host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->flag = tcph->res1;

				#ifdef DEBUG
					printk("%c%c\n", host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[0], host_ctrl->msg_queue[host_ctrl->msg_tail_i].msg.ipmsg->message[1]);
				#endif
			}
		}
		temp_ctrl = temp_ctrl->next;
	}
return;
}

#ifndef DEBUG
static
#endif
void pkt_router(struct sk_buff * skb) {

	if (gbusy == 1) {
		#ifdef DEBUG
			printk("pkt not processed\n");
		#endif
		return;
	}

	spin_lock_bh(&packet_lock);
	local_irq_disable();
	gbusy = 1;

	if (skb->protocol == __constant_htons(ETH_P_ARP)) {
		arp_rcv(skb);
	} else if (skb->protocol == __constant_htons(ETH_P_IP)) {
		ip_diff_rcv(skb);

		#ifdef WITH_TCP_UNIQUE
		if (skb->nh.iph->protocol == IPPROTO_TCP) {
			tcp_rcv(skb);
		}
		#endif
	}

	gbusy = 0;

	local_irq_enable();
	spin_unlock_bh(&packet_lock);
}


int init_module(void) {
	/* create procfile */
	struct proc_dir_entry * proc_ctrl_file;

	proc_ctrl_file = create_proc_entry(PROC_CTRL_FILE_NAME, 0644, NULL);
	proc_ctrl_file->read_proc = read_ctrl_proc;
	proc_ctrl_file->write_proc = write_ctrl_proc;

	proc_collection_dir = proc_mkdir(PROC_COLLECTION_DIR_NAME,  NULL);
	
	spin_lock_init(&proc_lock);
	spin_lock_init(&packet_lock);

	register_recv_packet_interceptor(pkt_router);

	printk("IPD Receive module loaded successfully\n");

	gbusy = 0;
	return 0;// ret;
}

void cleanup_module(void) {
	remove_proc_entry(PROC_CTRL_FILE_NAME, NULL);
	delete_all_listeners();
	remove_proc_entry(PROC_COLLECTION_DIR_NAME, NULL);
	spin_lock_bh(&packet_lock);
	local_irq_disable();
	register_recv_packet_interceptor(NULL);
	local_irq_enable();
	spin_unlock_bh(&packet_lock);
	printk("IPD Receive module unloaded\n");
}
